<?php
/**
 * Diagnóstico da chave privada do cliente
 * Verifica e corrige formato PEM na base de dados
 */

// Define BASEPATH para evitar bloqueio do CodeIgniter
define('BASEPATH', __DIR__ . '/app/');

require_once('app/config/database.php');

$conn = @new mysqli(DB_HOST, DB_USER, DB_PASS, DB_NAME);

if ($conn->connect_error) {
    die("❌ Erro de conexão: " . $conn->connect_error);
}

echo "<h2>Diagnóstico da Chave Privada do Cliente</h2>";

// Buscar chave privada
$result = $conn->query("SELECT private FROM tec_settings WHERE setting_id = 1");

if (!$result || $result->num_rows === 0) {
    die("❌ Nenhum registo encontrado em tec_settings (setting_id=1)");
}

$row = $result->fetch_assoc();
$privateKey = $row['private'];

echo "<h3>Estado Atual:</h3>";
echo "<p><strong>Tamanho:</strong> " . strlen($privateKey) . " caracteres</p>";
echo "<p><strong>Começa com:</strong> " . substr($privateKey, 0, 50) . "...</p>";
echo "<p><strong>Termina com:</strong> ..." . substr($privateKey, -50) . "</p>";

// Verificar formato
$hasBegin = strpos($privateKey, '-----BEGIN') !== false;
$hasEnd = strpos($privateKey, '-----END') !== false;
$hasNewlines = strpos($privateKey, "\n") !== false;

echo "<h3>Verificações:</h3>";
echo "<p>✓ Contém BEGIN: " . ($hasBegin ? '✅ Sim' : '❌ Não') . "</p>";
echo "<p>✓ Contém END: " . ($hasEnd ? '✅ Sim' : '❌ Não') . "</p>";
echo "<p>✓ Contém quebras de linha (\\n): " . ($hasNewlines ? '✅ Sim' : '❌ Não') . "</p>";

// Testar com OpenSSL
echo "<h3>Teste OpenSSL:</h3>";
$pkey = @openssl_pkey_get_private($privateKey);
if ($pkey) {
    echo "<p>✅ Chave VÁLIDA - OpenSSL conseguiu ler!</p>";
    openssl_free_key($pkey);
} else {
    $error = '';
    while ($e = openssl_error_string()) {
        $error .= $e . " | ";
    }
    echo "<p>❌ Chave INVÁLIDA - Erro OpenSSL: <code>" . htmlspecialchars($error) . "</code></p>";
    
    // Tentar normalizar
    echo "<h3>Tentativa de Normalização:</h3>";
    
    $normalized = trim($privateKey);
    
    // Converter escapes literais
    $normalized = str_replace(array("\\r\\n", "\\n", "\\r"), "\n", $normalized);
    
    // Cortar lixo antes do BEGIN
    $pos = strpos($normalized, '-----BEGIN');
    if ($pos !== false && $pos > 0) {
        $normalized = substr($normalized, $pos);
    }
    
    // Forçar BEGIN/END em linhas separadas
    $normalized = preg_replace("/(-----BEGIN [^-]+-----)\s*/", "$1\n", $normalized);
    $normalized = preg_replace("/\s*(-----END [^-]+-----)/", "\n$1", $normalized);
    
    // Remover espaços no fim das linhas
    $normalized = preg_replace("/[ \t]+\n/", "\n", $normalized);
    
    if (substr($normalized, -1) !== "\n") {
        $normalized .= "\n";
    }
    
    // Testar normalizada
    $pkey2 = @openssl_pkey_get_private($normalized);
    if ($pkey2) {
        echo "<p>✅ Chave normalizada é VÁLIDA!</p>";
        openssl_free_key($pkey2);
        
        echo "<h3>🔧 AÇÃO NECESSÁRIA</h3>";
        echo "<p style='background:#fff3cd;padding:15px;border:2px solid #ffc107;color:#856404;font-size:16px;'>";
        echo "<strong>A chave privada precisa ser atualizada na base de dados.</strong><br>";
        echo "Clique no botão abaixo para guardar o formato correto.";
        echo "</p>";
        
        if (isset($_POST['fix'])) {
            $stmt = $conn->prepare("UPDATE tec_settings SET private = ? WHERE setting_id = 1");
            $stmt->bind_param('s', $normalized);
            
            if ($stmt->execute()) {
                echo "<div style='background:#d4edda;padding:20px;border:2px solid #28a745;color:#155724;margin:20px 0;'>";
                echo "<h2 style='margin:0 0 10px 0;'>✅ CHAVE CORRIGIDA COM SUCESSO!</h2>";
                echo "<p style='margin:0;'>A chave privada foi atualizada na base de dados.</p>";
                echo "<p style='margin:10px 0 0 0;'><strong>Próximo passo:</strong> Volte às Settings e clique em \"Consultar Séries (AGT) e Guardar\"</p>";
                echo "</div>";
                echo "<p><a href='settings' style='display:inline-block;padding:10px 20px;background:#007bff;color:white;text-decoration:none;border-radius:5px;'>➡️ Ir para Settings</a></p>";
            } else {
                echo "<p style='background:#f8d7da;padding:10px;border:1px solid #f5c6cb;color:#721c24;'>❌ Erro ao atualizar: " . $stmt->error . "</p>";
            }
        } else {
            echo '<form method="post" style="margin:20px 0;">';
            echo '<button type="submit" name="fix" value="1" style="padding:15px 30px;background:#28a745;color:white;border:none;cursor:pointer;font-size:18px;border-radius:5px;font-weight:bold;">✅ GUARDAR CHAVE CORRIGIDA AGORA</button>';
            echo '</form>';
        }
    } else {
        $error2 = '';
        while ($e = openssl_error_string()) {
            $error2 .= $e . " | ";
        }
        echo "<p>❌ Mesmo após normalização, chave continua inválida: <code>" . htmlspecialchars($error2) . "</code></p>";
        echo "<p><strong>A chave privada precisa ser regerada ou importada corretamente.</strong></p>";
    }
}

echo "<hr>";
echo "<h3>Visualização Completa (primeiros 500 caracteres):</h3>";
echo "<pre style='background:#f5f5f5;padding:10px;overflow:auto;'>" . htmlspecialchars(substr($privateKey, 0, 500)) . "...</pre>";

$conn->close();
