<?php
defined('BASEPATH') OR exit('No direct script access allowed');

class Agt extends CI_Controller
{
    public function __construct()
    {
        parent::__construct();
        $this->load->database();
        $this->load->helper(array('url'));
    }

    // Diagnóstico da venda (payload, keys e testes de assinatura)
    public function diagnose_sale($sale_id = null)
    {
        // Apenas utilizadores autenticados
        if (!$this->session->userdata('user_id')) {
            return $this->_json(array('ok'=>false,'http_code'=>403,'error'=>'Unauthorized'));
        }

        $sale_id = (int)$sale_id;
        if (!$sale_id) return $this->_json(array('ok'=>false,'http_code'=>400,'error'=>'sale_id inválido'));

        $cfg = $this->_agt_cfg();
        if (!$cfg['ok']) return $this->_json($cfg);

        $sale = $this->db->get_where('tec_sales', array('id'=>$sale_id), 1)->row();
        if (!$sale) return $this->_json(array('ok'=>false,'http_code'=>404,'error'=>'Venda não encontrada'));

        $items = $this->db->get_where('tec_sale_items', array('sale_id'=>$sale_id))->result();
        if (!$items) $items = array();

        // Gera payload como faria ao submeter
        $payload = $this->_build_registar_payload($cfg, $sale, $items);

        // Verifica chaves
        $clientKeyPath = isset($cfg['client_private_key_path']) ? $cfg['client_private_key_path'] : null;
        $softKeyPath = isset($cfg['software_private_key_path']) ? $cfg['software_private_key_path'] : null;

        $clientKeyExists = ($clientKeyPath && file_exists($clientKeyPath) && is_readable($clientKeyPath));
        $softKeyExists = ($softKeyPath && file_exists($softKeyPath) && is_readable($softKeyPath));

        // Teste rápido de assinatura JWS (não envia nada)
        $sign_test_client = '';
        $sign_test_software = '';
        try {
            if ($clientKeyExists) $sign_test_client = $this->_jws_sign_json_with_filekey(array('test'=>1), $clientKeyPath);
        } catch (Exception $e) { $sign_test_client = ''; }
        try {
            if ($softKeyExists) $sign_test_software = $this->_jws_sign_json_with_filekey(array('test'=>1), $softKeyPath);
        } catch (Exception $e) { $sign_test_software = ''; }

        return $this->_json(array(
            'ok' => true,
            'cfg' => $cfg,
            'sale' => array('id'=>$sale->id,'InvoiceNo'=>isset($sale->InvoiceNo)?$sale->InvoiceNo:null,'agt_request_id'=>isset($sale->agt_request_id)?$sale->agt_request_id:null,'agt_status'=>isset($sale->agt_status)?$sale->agt_status:null),
            'payload' => $payload,
            'client_key' => array('path'=>$clientKeyPath,'exists'=>$clientKeyExists,'sign_ok'=>($sign_test_client?true:false)),
            'software_key' => array('path'=>$softKeyPath,'exists'=>$softKeyExists,'sign_ok'=>($sign_test_software?true:false)),
        ));
    }

    

    /* =========================================================
     * TESTE RÁPIDO: /Agt/test_cfg
     * =======================================================*/
    public function test_cfg()
    {
        error_reporting(E_ALL);
        ini_set('display_errors', 1);

        $out = array(
            'ok' => true,
            'php' => PHP_VERSION,
            'openssl_loaded' => extension_loaded('openssl'),
            'curl_loaded' => extension_loaded('curl'),
        );

        $cfg = $this->_agt_cfg();
        $out['cfg'] = $cfg;

        $cacheDir = APPPATH . 'cache/agt_keys/';
        $out['cacheDir'] = $cacheDir;
        $out['cacheDir_exists'] = is_dir($cacheDir);
        $out['cacheDir_writable'] = is_writable($cacheDir);

        return $this->_json($out);
    }

    /* =========================================================
     * 1) REGISTAR FACTURA - /fe/v1/registarFactura
     * URL: /Agt/registar_factura/{sale_id}?send=1&debug=1
     * =======================================================*/
    public function registar_factura($sale_id = null)
    {
        $sale_id = (int)$sale_id;
        $send  = (int)$this->input->get('send');
        $debug = (int)$this->input->get('debug');

        if (!$sale_id) return $this->_json(array('ok'=>false,'http_code'=>400,'error'=>'sale_id inválido'));
        if (!$send) {
            return $this->_json(array('ok'=>true,'http_code'=>200,'payload'=>null,'debug_note'=>'Sem envio. Usa ?send=1'));
        }

        // Verificar se tipo de faturação é SAF-T AO (não envia para AGT)
        $settings = $this->db->get_where('settings', array('setting_id' => 1))->row();
        if ($settings && isset($settings->tipo_faturacao) && $settings->tipo_faturacao === 'SAF-T AO') {
            return $this->_json(array(
                'ok'=>true,
                'http_code'=>200,
                'skipped'=>true,
                'message'=>'Tipo de faturação é SAF-T AO. Envio para AGT desativado.'
            ));
        }

        // ✅ Verifica se é FP (Proforma) - se sim, converte para FT primeiro
        $sale = $this->db->get_where('tec_sales', array('id'=>$sale_id), 1)->row();
        if (!$sale) return $this->_json(array('ok'=>false,'http_code'=>404,'error'=>'Venda não encontrada em tec_sales'));

        $invoiceType = isset($sale->InvoiceType) ? strtoupper(trim($sale->InvoiceType)) : '';
        if ($invoiceType === 'FP') {
            // Converte FP para FT e usa o novo ID
            $converted = $this->_convert_fp_to_ft($sale_id);
            if (!$converted['ok']) {
                return $this->_json(array(
                    'ok'=>false,
                    'http_code'=>422,
                    'error'=>'Erro ao converter FP para FT: ' . $converted['error']
                ));
            }
            // Usa o novo sale_id da FT convertida
            $sale_id = $converted['new_sale_id'];
            $sale = $this->db->get_where('tec_sales', array('id'=>$sale_id), 1)->row();
        }

// ✅ MARCADOR: prova que a rota foi chamada
$u0 = $this->_update_sale_agt($sale_id, array(
    'agt_submitted_at' => date('Y-m-d H:i:s'),
    'agt_last_status'  => 'DISPARADO',
    'agt_status'       => 'DISPARADO',
    'agt_error_json'   => json_encode(array(
        'note' => 'Endpoint Agt/registar_factura foi chamado',
        'get'  => $this->input->get(),
    ), JSON_UNESCAPED_UNICODE),
));

        $cfg = $this->_agt_cfg();
        if (!$cfg['ok']) {
            // aqui NÃO existe $res ainda. Só devolve cfg e marca na venda.
            $this->_update_sale_agt($sale_id, array(
                'agt_submitted_at' => date('Y-m-d H:i:s'),
                'agt_last_status'  => 'ERRO_CFG',
                'agt_status'       => 'ERRO',
                'agt_error_json'   => json_encode($cfg, JSON_UNESCAPED_UNICODE),
            ));
            return $this->_json($cfg);
        }

        $sale = $this->db->get_where('tec_sales', array('id'=>$sale_id), 1)->row();
        if (!$sale) return $this->_json(array('ok'=>false,'http_code'=>404,'error'=>'Venda não encontrada em tec_sales'));

        $items = $this->db->get_where('tec_sale_items', array('sale_id'=>$sale_id))->result();
        if (!$items) $items = array();

        $payload = $this->_build_registar_payload($cfg, $sale, $items);

        // bloqueios do payload (se usares)
        if (is_array($payload) && isset($payload['_abort']) && $payload['_abort']) {
            $this->_update_sale_agt($sale_id, array(
                'agt_submitted_at'=> date('Y-m-d H:i:s'),
                'agt_last_status' => 'BLOQUEADO',
                'agt_status'      => 'ERRO',
                'agt_error_json'  => json_encode($payload, JSON_UNESCAPED_UNICODE)
            ));
            return $this->_json(array('ok'=>false,'http_code'=>422,'error'=>$payload['_error'],'details'=>$payload));
        }

        $url = rtrim($cfg['base_url'],'/') . '/fe/v1/registarFactura';
        $res = $this->_http_json('POST', $url, $payload, $cfg['username'], $cfg['password'], $debug);

        $now = date('Y-m-d H:i:s');
        $json = $this->_safe_json($res['body']);
        $requestID = isset($json['requestID']) ? trim((string)$json['requestID']) : '';

        // ✅ SEMPRE guarda o retorno bruto (SUCESSO ou ERRO)
        $this->_update_sale_agt($sale_id, array(
            'agt_submitted_at' => $now,
            'agt_error_json'   => json_encode(array(
                'http_code' => (int)$res['http_code'],
                'ok'        => (bool)$res['ok'],
                'body'      => (string)$res['body'],
                'json'      => $json,
                'curl_err'  => (string)$res['error'],
            ), JSON_UNESCAPED_UNICODE),
        ));
        // ✅ se veio requestID, marca submetido — polling fica para background/cron (sem bloquear POS)
        if ($requestID !== '') {
            $this->_update_sale_agt($sale_id, array(
                'agt_request_id'   => $requestID,
                'agt_last_status'  => 'SUBMITTED',
                'agt_status'       => 'ENVIADO'
            ));

            // Resposta imediata (não bloqueia para polling)
            $response = array(
                'ok' => true,
                'http_code' => (int)$res['http_code'],
                'payload' => $payload,
                'requestID' => $requestID,
                'agt_submit' => $json,
                'message' => 'Factura submetida à AGT. Estado será consultado em background.',
                'debug' => isset($res['_debug']) ? $res['_debug'] : null
            );

            // Envia resposta ao cliente
            $this->_json($response);

            // Se possível, finaliza a resposta ao cliente antes de continuar (FastCGI)
            if (function_exists('fastcgi_finish_request')) {
                @session_write_close();
                @fastcgi_finish_request();
            }

            // Envia resposta ao cliente já feita acima.
            // Se o ambiente suporta FastCGI, usamos shutdown function com sleep(10)
            // para fazer um pedido interno silencioso (não bloqueia o cliente).
            if (function_exists('fastcgi_finish_request')) {
                @session_write_close();
                @fastcgi_finish_request();

                $check_url = $this->config->site_url('Agt/obter_estado/' . $sale_id . '?send=1');
                register_shutdown_function(function($url) {
                    // espera 10 segundos antes de consultar o estado
                    sleep(10);
                    // faz um pedido HTTP interno ao próprio endpoint (silencioso)
                    $ch = curl_init($url);
                    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
                    curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 5);
                    curl_setopt($ch, CURLOPT_TIMEOUT, 15);
                    // supress warnings/errors
                    @curl_exec($ch);
                    @curl_close($ch);
                }, $check_url);

                return;
            }

            // Caso não haja suporte a fastcgi_finish_request (ex.: mod_php), não usamos sleep()
            // para evitar bloquear o processo. Em vez disso, enfileiramos uma tentativa para +10s.
            $enq = $this->_enqueue_agt($sale_id);
            if (isset($enq['ok']) && $enq['ok'] && isset($enq['id'])) {
                $next = date('Y-m-d H:i:s', time() + 10);
                $this->db->where('id', $enq['id'])->update('tec_agt_queue', array('next_attempt' => $next, 'updated_at' => date('Y-m-d H:i:s')));
            }

            return;
        }
        // ❌ sem requestID OU falha HTTP: enfileira para re-tentativa (fila DB)
        if (!$res['ok'] || $requestID === '') {
            $this->_update_sale_agt($sale_id, array(
                'agt_last_status'  => 'ERRO_SUBMIT',
                'agt_status'       => 'ERRO'
            ));

            $enq = $this->_enqueue_agt($sale_id);

            return $this->_json(array(
                'ok' => false,
                'http_code' => (int)$res['http_code'],
                'queued' => true,
                'queue_id' => isset($enq['id']) ? $enq['id'] : null,
                'error' => 'AGT não devolveu requestID ou erro de envio. Enfileirado para re-tentativa.',
                'payload' => $payload,
                'agt' => $json,
                'debug' => isset($res['_debug']) ? $res['_debug'] : null
            ));
        }
    }

    /* =========================================================
 * CLI: php index.php agt registar_factura_cli 276
 * Força send=1 e chama o registar_factura normal.
 * =======================================================*/
public function registar_factura_cli($sale_id = null)
{
    // Só permitir via CLI
    if (!is_cli()) show_404();

    $sale_id = (int)$sale_id;
    if (!$sale_id) {
        echo "sale_id inválido\n";
        return;
    }

    // Forçar envio
    $_GET['send'] = 1;
    $_GET['debug'] = 0;

    // Reutiliza a função normal (não duplicamos lógica)
    $this->registar_factura($sale_id);
}


    /* =========================================================
     * 2) OBTER ESTADO - /fe/v1/obterEstado
     * URL: /Agt/obter_estado/{sale_id}?send=1
     * =======================================================*/
    public function obter_estado($sale_id = null)
    {
        $sale_id = (int)$sale_id;
        $send  = (int)$this->input->get('send');
        $debug = (int)$this->input->get('debug');

        if (!$sale_id) return $this->_json(array('ok'=>false,'http_code'=>400,'error'=>'sale_id inválido'));
        if (!$send) return $this->_json(array('ok'=>true,'http_code'=>200,'payload'=>null,'debug_note'=>'Sem consulta. Usa ?send=1'));

        $cfg = $this->_agt_cfg();
        if (!$cfg['ok']) return $this->_json($cfg);

        $sale = $this->db->get_where('tec_sales', array('id'=>$sale_id), 1)->row();
        if (!$sale) return $this->_json(array('ok'=>false,'http_code'=>404,'error'=>'Venda não encontrada'));

        // ✅ Se já está VALIDA, não consulta de novo
        $currentStatus = isset($sale->agt_status) ? strtoupper(trim($sale->agt_status)) : '';
        if ($currentStatus === 'VALIDA') {
            return $this->_json(array(
                'ok'=>true,
                'http_code'=>200,
                'skipped'=>true,
                'message'=>'Factura já está VALIDA, não é necessário consultar',
                'sale_id'=>$sale_id,
                'agt_status'=>$sale->agt_status
            ));
        }

        $requestID = isset($sale->agt_request_id) ? trim((string)$sale->agt_request_id) : '';
        if ($requestID === '') {
            return $this->_json(array(
                'ok'=>false,
                'http_code'=>422,
                'error'=>'Sem agt_request_id. Envia primeiro a factura.',
                'sale_id'=>$sale_id,
                'agt_status'=>isset($sale->agt_status) ? $sale->agt_status : null
            ));
        }

        $consulta = $this->_consultar_estado_agora($sale_id, $requestID, $cfg, $debug);

        return $this->_json(array(
            'ok' => (bool)$consulta['ok'],
            'http_code' => (int)$consulta['http_code'],
            'requestID' => $requestID,
            'agt_estado' => $consulta
        ));
    }

    /* ===================== CONSULTA AUTOMÁTICA ===================== */
    private function _consultar_estado_agora($sale_id, $requestID, $cfg, $debug = 0)
    {
        $payload = $this->_build_obter_estado_payload($cfg, $requestID);
        $url = rtrim($cfg['base_url'],'/') . '/fe/v1/obterEstado';

        $res = $this->_http_json('POST', $url, $payload, $cfg['username'], $cfg['password'], $debug);
        $json = $this->_safe_json($res['body']);

        // Verifica se há erro na resposta AGT (requestErrorList)
        $errorId = '';
        $errorDesc = '';
        if (isset($json['requestErrorList'][0]['idError'])) {
            $errorId = (string)$json['requestErrorList'][0]['idError'];
        }
        if (isset($json['requestErrorList'][0]['descriptionError'])) {
            $errorDesc = (string)$json['requestErrorList'][0]['descriptionError'];
        }

        // tenta ler status do documento
        $status = '';
        if (isset($json['documentStatusList'][0]['documentStatus'])) {
            $status = (string)$json['documentStatusList'][0]['documentStatus']; // V / I / etc
        }

        // Verifica erros específicos do documento (errorList - ex: E40)
        $docErrorId = '';
        $docErrorDesc = '';
        if (isset($json['documentStatusList'][0]['errorList'][0]['idError'])) {
            $docErrorId = (string)$json['documentStatusList'][0]['errorList'][0]['idError'];
        }
        if (isset($json['documentStatusList'][0]['errorList'][0]['descriptionError'])) {
            $docErrorDesc = (string)$json['documentStatusList'][0]['errorList'][0]['descriptionError'];
        }

        // mapeia para colunas: VALIDA quando aprovada, ENVIADA para tudo o resto (incluindo I - inválida)
        $agt_last = 'ENVIADA';
        $agt_stat = 'ENVIADA';

        if ($status === 'V') {
            $agt_last = 'VALIDA';
            $agt_stat = 'VALIDA';
        }
        // Nota: Se status === 'I' (INVÁLIDA), mantém ENVIADA e regista o erro em agt_error_json

        // ✅ SEMPRE guardar tudo (SUCESSO ou ERRO)
        $this->_update_sale_agt($sale_id, array(
            'agt_last_status' => $agt_last,
            'agt_status'      => $agt_stat,
            'agt_error_json'  => json_encode(array(
                'http_code'     => (int)$res['http_code'],
                'ok'            => (bool)$res['ok'],
                'requestID'     => (string)$requestID,
                'status'        => (string)$status,
                'errorId'       => $errorId,
                'errorDesc'     => $errorDesc,
                'docErrorId'    => $docErrorId,
                'docErrorDesc'  => $docErrorDesc,
                'json'          => $json,
                'body'          => (string)$res['body'],
                'curl_err'      => (string)$res['error'],
            ), JSON_UNESCAPED_UNICODE)
        ));

        // ✅ Se for VALIDA, atualizar tec_settings.agt_status para VALIDA
        if ($status === 'V' && $agt_last === 'VALIDA') {
            $this->db->update('tec_settings', array('agt_status' => 'VALIDA'));
        }

        return array(
            'ok' => (bool)$res['ok'] && $errorId === '' && $docErrorId === '',
            'http_code' => (int)$res['http_code'],
            'payload' => $payload,
            'status' => $status,
            'errorId' => $errorId,
            'errorDesc' => $errorDesc,
            'docErrorId' => $docErrorId,
            'docErrorDesc' => $docErrorDesc,
            'json' => $json,
            'raw_body' => $res['body']
        );
    }

    /* ===================== BUILDERS ===================== */

    private function _build_obter_estado_payload($cfg, $requestID)
    {
        $taxNo = $cfg['taxRegistrationNumber'];

        $forSign = array(
            'requestID' => (string)$requestID,
            'taxRegistrationNumber' => (string)$taxNo
        );

        $jws = $this->_jws_sign_json_with_filekey($forSign, $cfg['client_private_key_path']);

        $softwareInfoDetail = array(
            'productId' => (string)$cfg['productId'],
            'productVersion' => (string)$cfg['productVersion'],
            'softwareValidationNumber' => (string)$cfg['softwareValidationNumber'],
        );

        // software assina com chave do produtor/software
        $jwsSoftwareSignature = $this->_jws_sign_json_with_filekey($softwareInfoDetail, $cfg['software_private_key_path']);

        return array(
            'schemaVersion' => '1.0',
            'taxRegistrationNumber' => (string)$taxNo,
            'softwareInfo' => array(
                'softwareInfoDetail' => $softwareInfoDetail,
                'jwsSoftwareSignature' => $jwsSoftwareSignature,
            ),
            'requestID' => (string)$requestID,
            'jwsSignature' => $jws,
        );
    }

    private function _build_registar_payload($cfg, $sale, $items)
    {
        $submissionUUID = $this->_uuid_v4();
        $taxNo = $cfg['taxRegistrationNumber'];

        // Precarrega códigos/nome de produtos e códigos de imposto para evitar inconsistências
        $productMap = array();
        $taxMap = array();
        $productIds = array();
        $taxIds = array();

        foreach ($items as $it) {
            if (isset($it->product_id) && (int)$it->product_id > 0) {
                $productIds[] = (int)$it->product_id;
            }
            if (isset($it->tax_id) && (int)$it->tax_id > 0) {
                $taxIds[] = (int)$it->tax_id;
            }
        }

        if (!empty($productIds)) {
            $productIds = array_values(array_unique($productIds));
            $qProd = $this->db->select('id, code, name')
                ->where_in('id', $productIds)
                ->get('tec_products');
            foreach ($qProd->result() as $p) {
                $productMap[(int)$p->id] = array(
                    'code' => ($p->code !== '' && $p->code !== null) ? trim((string)$p->code) : (string)$p->id,
                    'name' => ($p->name !== '' && $p->name !== null) ? trim((string)$p->name) : 'Produto',
                );
            }
        }

        if (!empty($taxIds)) {
            $taxIds = array_values(array_unique($taxIds));
            $qTax = $this->db->select('id, code, tax, tax_type, tax_code, reason')
                ->where_in('id', $taxIds)
                ->get('tec_tax');
            foreach ($qTax->result() as $t) {
                $taxMap[(int)$t->id] = array(
                    'code' => trim((string)$t->code),
                    'tax_code' => ($t->tax_code !== '' && $t->tax_code !== null) ? trim((string)$t->tax_code) : 'ISE',
                    'rate' => isset($t->tax) ? (float)$t->tax : 0,
                    'type' => ($t->tax_type !== '' && $t->tax_type !== null) ? trim((string)$t->tax_type) : 'IVA',
                    'reason' => trim((string)$t->reason),
                );
            }
        }

        $docType = strtoupper(trim((string)$sale->InvoiceType));
        if ($docType === '') $docType = 'FT';

        $invNo = trim((string)$sale->InvoiceNo);
        $invNo = preg_replace('/^(FP|FA|FT|FR|FG|GF|AC|AR|TV|RC|RG|RE|ND|NC|AF|RP|RA|CS|LD)\s+/i', '', $invNo);
        $docNo = trim($docType . ' ' . $invNo);

        $docDate = date('Y-m-d');
        if (isset($sale->date) && trim((string)$sale->date) !== '') {
            $ts = strtotime((string)$sale->date);
            if ($ts) $docDate = date('Y-m-d', $ts);
        }

        $systemEntry = date('Y-m-d\TH:i:s');

        $customerCountry = 'AO';
        $customerTaxID   = '999999999';
        $companyName     = !empty($sale->customer_name) ? (string)$sale->customer_name : 'Cliente final';

        $lines = array();
        $netTotal = 0;
        $taxPayable = 0;
        $lineNo = 1;

        foreach ($items as $it) {
            $qty = isset($it->quantity) ? (float)$it->quantity : 1;
            $unitPrice = isset($it->unit_price) ? (float)$it->unit_price : 0;
            // Usar net_unit_price se disponível (já com desconto aplicado)
            $netUnitPrice = isset($it->net_unit_price) ? (float)$it->net_unit_price : $unitPrice;
            $credit = round($qty * $netUnitPrice, 2);

            // Imposto: usar o valor já calculado e salvo na base de dados
            $taxPercentage = 0;
            $taxCode = 'ISE';
            $taxType = 'IVA';
            $taxExemptionCode = 'M02';
            
            // Usar item_tax da base de dados (já calculado corretamente no Pos.php)
            $lineTax = isset($it->item_tax) ? (float)$it->item_tax : 0;

            if (isset($it->tax_id) && (int)$it->tax_id > 0) {
                $tid = (int)$it->tax_id;
                if (isset($taxMap[$tid])) {
                    $tx = $taxMap[$tid];
                    $taxPercentage = $tx['rate'];
                    $taxType = $tx['type'] !== '' ? $tx['type'] : 'IVA';
                    $taxCode = $tx['tax_code'] !== '' ? $tx['tax_code'] : 'ISE';
                    $taxExemptionCode = ($taxPercentage > 0) ? '' : 'M02';
                } else {
                    // Fallback se não encontrou no mapa
                    $taxPercentage = 0;
                    $taxCode = 'ISE';
                    $taxExemptionCode = 'M02';
                }
            } elseif (isset($it->tax) && is_numeric($it->tax)) {
                $taxPercentage = (float)$it->tax;
                $taxType = 'IVA';
                
                // Fallback se não houver tax_id
                if ($taxPercentage >= 14) {
                    $taxCode = 'NOR';
                    $taxExemptionCode = '';
                } elseif ($taxPercentage > 0 && $taxPercentage < 14) {
                    $taxCode = 'RED';
                    $taxExemptionCode = '';
                } else {
                    $taxCode = 'ISE';
                    $taxExemptionCode = 'M02';
                }
            }

            $taxObj = array(
                'taxType' => $taxType,
                'taxCountryRegion' => 'AO',
                'taxCode' => $taxCode,
                'taxPercentage' => (float)$taxPercentage,
                'taxContribution' => (float)$lineTax,
            );
            if ($taxExemptionCode !== '') {
                $taxObj['taxExemptionCode'] = $taxExemptionCode;
            }

            $productCode = '';
            if (isset($it->product_id) && (int)$it->product_id > 0) {
                $pid = (int)$it->product_id;
                if (isset($productMap[$pid]) && $productMap[$pid]['code'] !== '') {
                    $productCode = $productMap[$pid]['code'];
                } else {
                    $productCode = (string)$pid;
                }
            } elseif (isset($it->product_code) && trim((string)$it->product_code) !== '') {
                $productCode = (string)$it->product_code;
            } else {
                $productCode = 'P' . (string)$lineNo;
            }

            $productDesc = '';
            if (isset($it->product_id) && (int)$it->product_id > 0) {
                $pid = (int)$it->product_id;
                if (isset($productMap[$pid]) && $productMap[$pid]['name'] !== '') {
                    $productDesc = $productMap[$pid]['name'];
                } else {
                    $productDesc = 'Produto';
                }
            } elseif (isset($it->product_name) && trim((string)$it->product_name) !== '') {
                $productDesc = (string)$it->product_name;
            } elseif (isset($it->productDescription) && trim((string)$it->productDescription) !== '') {
                $productDesc = (string)$it->productDescription;
            } else {
                $productDesc = 'Produto';
            }

            $lines[] = array(
                'lineNumber' => (int)$lineNo,
                'productCode' => (string)$productCode,
                'productDescription' => (string)$productDesc,
                'quantity' => (float)$qty,
                'unitOfMeasure' => 'UN',
                'unitPrice' => (float)$netUnitPrice,
                'unitPriceBase' => (float)$unitPrice,
                'debitAmount' => (float)0,
                'creditAmount' => (float)$credit,
                'taxes' => array($taxObj),
                'settlementAmount' => (float)0,
            );

            $netTotal += $credit;
            $taxPayable += $lineTax;
            $lineNo++;
        }

        // Usar os totais já calculados e salvos na venda (tec_sales)
        // Se disponíveis, caso contrário usa os somados dos itens
        $finalNetTotal = isset($sale->total) && $sale->total > 0 
            ? (float)$sale->total 
            : (float)$netTotal;
        
        $finalTaxPayable = isset($sale->product_tax) && $sale->product_tax > 0 
            ? (float)$sale->product_tax 
            : (float)$taxPayable;
        
        $finalGrossTotal = isset($sale->grand_total) && $sale->grand_total > 0 
            ? (float)$sale->grand_total 
            : round($finalNetTotal + $finalTaxPayable, 2);

        // ✅ ESTRUTURA PARA ASSINATURA conforme documentação AGT
        // Apenas campos essenciais (SEM documentStatus, systemEntryDate, lines)
        // https://portaldoparceiro.hml.minfin.gov.ao/doc-agt/faturacao-electronica/1/estrutura.html#_assinatura_jwsdocumentsignature
        $docForSign = array(
            'documentNo' => (string)$docNo,
            'taxRegistrationNumber' => (string)$taxNo,
            'documentType' => (string)$docType,
            'documentDate' => (string)$docDate,
            'customerTaxID' => (string)$customerTaxID,
            'customerCountry' => (string)$customerCountry,
            'companyName' => (string)$companyName,
            'documentTotals' => array(
                'taxPayable' => (float)$finalTaxPayable,
                'netTotal' => (float)$finalNetTotal,
                'grossTotal' => (float)$finalGrossTotal
            ),
        );

        // DEBUG: Salvar documento antes de assinar
        $cacheDir = APPPATH . 'cache/';
        @file_put_contents($cacheDir . 'agt_doc_before_sign.json', json_encode($docForSign, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE));
        
        // DEBUG: Salvar exatamente o que será assinado (sem pretty print, como o JWT fará)
        $jsonForSignature = json_encode($docForSign, JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE);
        @file_put_contents($cacheDir . 'agt_doc_signed_payload.json', $jsonForSignature);

        // Assinar APENAS os campos essenciais
        $jwsDoc = $this->_jws_sign_json_with_filekey($docForSign, $cfg['client_private_key_path']);

        // ✅ DOCUMENTO COMPLETO para envio (com TODOS os campos)
        $doc = array(
            'documentNo' => (string)$docNo,
            'taxRegistrationNumber' => (string)$taxNo,
            'documentStatus' => 'N',
            'documentDate' => (string)$docDate,
            'documentType' => (string)$docType,
            'systemEntryDate' => (string)$systemEntry,
            'customerCountry' => (string)$customerCountry,
            'customerTaxID' => (string)$customerTaxID,
            'companyName' => (string)$companyName,
            'lines' => $lines,
            'documentTotals' => array(
                'taxPayable' => (float)$finalTaxPayable,
                'netTotal' => (float)$finalNetTotal,
                'grossTotal' => (float)$finalGrossTotal
            ),
            'jwsDocumentSignature' => $jwsDoc,
        );

        // DEBUG: Salvar documento final (com assinatura)
        @file_put_contents($cacheDir . 'agt_doc_after_sign.json', json_encode($doc, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE));
        
        // DEBUG: Decodificar JWT para verificar
        $jwtParts = explode('.', $jwsDoc);
        if (count($jwtParts) === 3) {
            $payloadDecoded = json_decode($this->_b64url_decode($jwtParts[1]), true);
            @file_put_contents($cacheDir . 'agt_jwt_payload_decoded.json', json_encode($payloadDecoded, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE));
        }

        $submissionForSign = array(
            'submissionUUID' => (string)$submissionUUID,
            'taxRegistrationNumber' => (string)$taxNo
        );
        $jwsSubmission = $this->_jws_sign_json_with_filekey($submissionForSign, $cfg['client_private_key_path']);

        $softwareInfoDetail = array(
            'productId' => (string)$cfg['productId'],
            'productVersion' => (string)$cfg['productVersion'],
            'softwareValidationNumber' => (string)$cfg['softwareValidationNumber'],
        );
        $jwsSoftwareSignature = $this->_jws_sign_json_with_filekey($softwareInfoDetail, $cfg['software_private_key_path']);

        return array(
            'schemaVersion' => '1.0',
            'submissionUUID' => (string)$submissionUUID,
            'taxRegistrationNumber' => (string)$taxNo,
            'submissionTimeStamp' => gmdate('Y-m-d\TH:i:s\Z'),
            'softwareInfo' => array(
                'softwareInfoDetail' => $softwareInfoDetail,
                'jwsSoftwareSignature' => $jwsSoftwareSignature,
            ),
            'jwsSignature' => $jwsSubmission,
            'numberOfEntries' => 1,
            'documents' => array($doc),
        );
    }

    /* ===================== CONFIG (BD) ===================== */

    private function _agt_cfg()
    {
        if (!extension_loaded('openssl')) {
            return array('ok'=>false,'http_code'=>500,'error'=>'Extensão OpenSSL não está carregada.');
        }
        if (!extension_loaded('curl')) {
            return array('ok'=>false,'http_code'=>500,'error'=>'Extensão cURL não está carregada.');
        }

        // settings (setting_id = 1)
        $q = $this->db->get_where('tec_settings', array('setting_id' => 1), 1);
        if (!$q) {
            return array('ok'=>false,'http_code'=>500,'error'=>'Falha na query tec_settings.', 'db_error'=>$this->db->error());
        }

        $st = $q->row();
        if (!$st) {
            return array('ok'=>false,'http_code'=>500,'error'=>'Config não encontrada em tec_settings (setting_id=1).');
        }

        $get = function($row, $keys){
            foreach ($keys as $k){
                if (isset($row->$k) && trim((string)$row->$k) !== '') return trim((string)$row->$k);
            }
            return '';
        };

        $base_url = $get($st, array('agt_base_url'));
        $username = $get($st, array('agt_username'));
        $password = $get($st, array('agt_password'));

        if ($base_url === '') $base_url = 'https://sifphml.minfin.gov.ao/sigt';

        $taxNo  = $get($st, array('TaxRegistrationNumber','taxRegistrationNumber'));
        $productId = $get($st, array('ProductID','ProductId','productId'));
        $productVersion = $get($st, array('ProductVersion','productVersion'));
        $softwareValidationNumber = $get($st, array('SoftwareValidationNumber','softwareValidationNumber'));

        if ($base_url === '' || $username === '' || $password === '') {
            return array('ok'=>false,'http_code'=>500,'error'=>'Campos AGT em falta: agt_base_url / agt_username / agt_password.');
        }

        if ($taxNo === '' || $productId === '' || $productVersion === '' || $softwareValidationNumber === '') {
            return array('ok'=>false,'http_code'=>500,'error'=>'Campos do software em falta: TaxRegistrationNumber / ProductID / ProductVersion / SoftwareValidationNumber.');
        }

        // chave cliente: tec_settings.private
        $clientPemRaw = $get($st, array('private'));
        if ($clientPemRaw === '') {
            return array('ok'=>false,'http_code'=>500,'error'=>'CHAVE CLIENTE vazia em tec_settings.private');
        }
        $clientPem = $this->_normalize_pem($clientPemRaw);

        $chkClient = $this->_assert_private_key_valid($clientPem, 'CHAVE CLIENTE');
        if (!$chkClient['ok']) return $chkClient;
        $clientPem = $chkClient['pem'];

        // chave software/produtor: tec_signkey.private
        $sk = $this->db->get_where('tec_signkey', array('id'=>1), 1)->row();
        if (!$sk || !isset($sk->private) || trim((string)$sk->private) === '') {
            return array('ok'=>false,'http_code'=>500,'error'=>'CHAVE SOFTWARE/PRODUTOR não encontrada em tec_signkey.private (id=1).');
        }
        $softPem = $this->_normalize_pem((string)$sk->private);

        $chkSoft = $this->_assert_private_key_valid($softPem, 'CHAVE SOFTWARE/PRODUTOR');
        if (!$chkSoft['ok']) return $chkSoft;
        $softPem = $chkSoft['pem'];

        // cache para ficheiros PEM
        $cacheDir = APPPATH . 'cache/agt_keys/';
        if (!is_dir($cacheDir)) @mkdir($cacheDir, 0700, true);
        if (!is_dir($cacheDir) || !is_writable($cacheDir)) {
            return array('ok'=>false,'http_code'=>500,'error'=>'Sem permissão para escrever em: '.$cacheDir);
        }

        $clientKeyPath = $cacheDir . 'cliente_private_key.pem';
        $softKeyPath   = $cacheDir . 'software_private_key.pem';

        if (@file_put_contents($clientKeyPath, $clientPem) === false) {
            return array('ok'=>false,'http_code'=>500,'error'=>'Falha ao gravar chave cliente em: '.$clientKeyPath);
        }
        if (@file_put_contents($softKeyPath, $softPem) === false) {
            return array('ok'=>false,'http_code'=>500,'error'=>'Falha ao gravar chave software em: '.$softKeyPath);
        }

        @chmod($clientKeyPath, 0600);
        @chmod($softKeyPath, 0600);

        return array(
            'ok' => true,
            'base_url' => $base_url,
            'username' => $username,
            'password' => $password,
            'taxRegistrationNumber' => $taxNo,
            'productId' => $productId,
            'productVersion' => $productVersion,
            'softwareValidationNumber' => $softwareValidationNumber,
            'client_private_key_path' => $clientKeyPath,
            'software_private_key_path' => $softKeyPath,
        );
    }

    /* ===================== PEM HELPERS ===================== */

    private function _normalize_pem($pem)
    {
        $pem = trim((string)$pem);

        // converte "\r\n" e "\n" literais em quebras reais
        $pem = str_replace(array("\\r\\n", "\\n", "\\r"), "\n", $pem);

        // corta lixo antes do BEGIN
        $pos = strpos($pem, '-----BEGIN');
        if ($pos !== false && $pos > 0) $pem = substr($pem, $pos);

        // Extrai o header, corpo e footer
        if (preg_match('/(-----BEGIN [^-]+-----)(.*?)(-----END [^-]+-----)/s', $pem, $matches)) {
            $header = trim($matches[1]);
            $body = $matches[2];
            $footer = trim($matches[3]);
            
            // Remove todas quebras de linha e espaços do corpo
            $body = preg_replace('/\s+/', '', $body);
            
            // Divide em linhas de 64 caracteres
            $body = chunk_split($body, 64, "\n");
            $body = trim($body);
            
            // Reconstrói a chave
            $pem = $header . "\n" . $body . "\n" . $footer . "\n";
        } else {
            // Fallback para normalização simples
            $pem = preg_replace("/(-----BEGIN [^-]+-----)\s*/", "$1\n", $pem);
            $pem = preg_replace("/\s*(-----END [^-]+-----)/", "\n$1", $pem);
            $pem = preg_replace("/[ \t]+\n/", "\n", $pem);
            if (substr($pem, -1) !== "\n") $pem .= "\n";
        }

        return $pem;
    }

    private function _assert_private_key_valid($pem, $label)
    {
        $pem = $this->_normalize_pem($pem);

        $pkey = @openssl_pkey_get_private($pem);
        if (!$pkey) {
            $err = '';
            while ($e = openssl_error_string()) { $err .= $e . ' | '; }
            $err = trim($err, " |");

            return array(
                'ok'=>false,
                'http_code'=>500,
                'error'=> $label.' inválida (openssl_pkey_get_private falhou).',
                'openssl_error' => $err
            );
        }
        openssl_free_key($pkey);

        return array('ok'=>true,'http_code'=>200,'pem'=>$pem);
    }

    /* ===================== HTTP ===================== */

    private function _http_json($method, $url, $payload, $user, $pass, $debug)
    {
        // CRÍTICO: usar mesmas flags que na assinatura JWT!
        $body = json_encode($payload, JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE);
        
        // DEBUG: Salvar exatamente o que está sendo enviado via HTTP
        $cacheDir = APPPATH . 'cache/';
        @file_put_contents($cacheDir . 'agt_http_sent_body.json', $body);
        @file_put_contents($cacheDir . 'agt_http_sent_payload.json', json_encode($payload, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE));

        $headers = array(
            'Content-Type: application/json; charset=utf-8',
            'Accept: application/json',
        );

        $ch = curl_init($url);
        curl_setopt_array($ch, array(
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_HTTPAUTH => CURLAUTH_BASIC,
            CURLOPT_USERPWD => $user . ':' . $pass,
            CURLOPT_HTTPHEADER => $headers,
            CURLOPT_TIMEOUT => 60,
            CURLOPT_SSL_VERIFYPEER => false,
            CURLOPT_SSL_VERIFYHOST => false,
        ));

        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, $body);

        $respBody = curl_exec($ch);
        $code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $err = curl_error($ch);
        $info = curl_getinfo($ch);
        curl_close($ch);

        $out = array(
            'ok' => ($code >= 200 && $code < 300),
            'http_code' => (int)$code,
            'body' => $respBody,
            'error' => $err,
        );

        if ($debug) {
            $out['_debug'] = array(
                'url' => $url,
                'used_basic_auth' => true,
                'sent_user' => $user,
                'payload' => $payload,
                'curl_info' => $info,
            );
        }

        return $out;
    }

    /* ===================== JWS ===================== */

    private function _jws_sign_json_with_filekey($payloadArray, $privateKeyPath)
    {
        $pem = @file_get_contents($privateKeyPath);
        if (!$pem) return '';
        return $this->_jws_sign_json_with_pemkey($payloadArray, $pem);
    }

    private function _jws_sign_json_with_pemkey($payloadArray, $privateKeyPem)
{
    $header = array('alg' => 'RS256', 'typ' => 'JWT');

    // IMPORTANTE: não escapar "/" nem unicode
    $jsonFlags = JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE;

    $h = $this->_b64url(json_encode($header, $jsonFlags));
    $p = $this->_b64url(json_encode($payloadArray, $jsonFlags));
    $data = $h . '.' . $p;

    // Normaliza formato PEM (corrige quebras de linha)
    $privateKeyPem = $this->_normalize_pem($privateKeyPem);

    $pkey = openssl_pkey_get_private($privateKeyPem);
    if (!$pkey) {
        $err = openssl_error_string();
        log_message('error', 'CHAVE CLIENTE inválida (openssl_pkey_get_private falhou). OpenSSL: ' . $err);
        return '';
    }

    $sig = '';
    openssl_sign($data, $sig, $pkey, OPENSSL_ALGO_SHA256);
    openssl_free_key($pkey);

    return $data . '.' . $this->_b64url($sig);
}


    private function _b64url($data)
    {
        $b64 = base64_encode($data);
        return rtrim(strtr($b64, '+/', '-_'), '=');
    }
    
    private function _b64url_decode($data)
    {
        $remainder = strlen($data) % 4;
        if ($remainder) {
            $padlen = 4 - $remainder;
            $data .= str_repeat('=', $padlen);
        }
        return base64_decode(strtr($data, '-_', '+/'));
    }

    private function _uuid_v4()
    {
        $data = openssl_random_pseudo_bytes(16);
        $data[6] = chr((ord($data[6]) & 0x0f) | 0x40);
        $data[8] = chr((ord($data[8]) & 0x3f) | 0x80);
        return vsprintf('%s%s-%s-%s-%s-%s%s%s', str_split(bin2hex($data), 4));
    }

    /* ===================== DB UPDATE ===================== */

 private function _update_sale_agt($sale_id, $data)
{
    if (!is_array($data) || empty($data)) return array('ok'=>false,'reason'=>'data_empty');

    $sale_id = (int)$sale_id;

    // 1) tenta por "id"
    $this->db->where('id', $sale_id);
    $this->db->update('tec_sales', $data);
    $err1 = $this->db->error();
    $aff1 = (int)$this->db->affected_rows();
    $q1   = $this->db->last_query();

    if ($err1['code'] == 0 && $aff1 > 0) {
        return array(
            'ok' => true,
            'matched_by' => 'id',
            'sale_id' => $sale_id,
            'affected_rows' => $aff1,
            'db_error' => $err1,
            'last_query' => $q1,
        );
    }

    // 2) fallback: tenta por "sale_id" (muito comum em alguns POS)
    $this->db->where('sale_id', $sale_id);
    $this->db->update('tec_sales', $data);
    $err2 = $this->db->error();
    $aff2 = (int)$this->db->affected_rows();
    $q2   = $this->db->last_query();

    return array(
        'ok' => ($err2['code'] == 0 && $aff2 > 0),
        'matched_by' => ($aff2 > 0 ? 'sale_id' : 'none'),
        'sale_id' => $sale_id,
        'affected_rows_id' => $aff1,
        'affected_rows_sale_id' => $aff2,
        'db_error_id' => $err1,
        'db_error_sale_id' => $err2,
        'last_query_id' => $q1,
        'last_query_sale_id' => $q2,
    );
}

    /* ===================== AGT QUEUE (DB) ===================== */

    private function _ensure_queue_table_exists()
    {
        $sql = "CREATE TABLE IF NOT EXISTS `tec_agt_queue` (
            `id` INT UNSIGNED NOT NULL AUTO_INCREMENT,
            `sale_id` INT NOT NULL,
            `attempts` INT NOT NULL DEFAULT 0,
            `status` VARCHAR(20) NOT NULL DEFAULT 'pending',
            `last_error` TEXT NULL,
            `next_attempt` DATETIME NULL,
            `created_at` DATETIME NOT NULL,
            `updated_at` DATETIME NULL,
            PRIMARY KEY (`id`),
            INDEX (`sale_id`),
            INDEX (`status`),
            INDEX (`next_attempt`)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";

        $this->db->query($sql);
    }

    private function _enqueue_agt($sale_id)
    {
        $sale_id = (int)$sale_id;
        if ($sale_id <= 0) return array('ok'=>false,'error'=>'invalid_sale_id');

        $this->_ensure_queue_table_exists();

        // evita duplicados se já houver pendente
        $row = $this->db->get_where('tec_agt_queue', array('sale_id'=>$sale_id,'status'=>'pending'))->row();
        if ($row) {
            return array('ok'=>true,'id'=>$row->id,'existing'=>true);
        }

        $now = date('Y-m-d H:i:s');
        $data = array(
            'sale_id' => $sale_id,
            'attempts' => 0,
            'status' => 'pending',
            'last_error' => null,
            'next_attempt' => $now, // tentar de imediato (mas em background/cron)
            'created_at' => $now,
            'updated_at' => $now,
        );

        $this->db->insert('tec_agt_queue', $data);
        $id = $this->db->insert_id();
        return array('ok'=>true,'id'=>$id,'existing'=>false);
    }

    // Processa fila: php index.php agt process_queue [limit]
    public function process_queue($limit = 50)
    {
        // permite via CLI ou via web com ?send=1
        if (!is_cli()) {
            $send = (int)$this->input->get('send');
            if (!$send) return $this->_json(array('ok'=>false,'error'=>'use ?send=1 or run via CLI'));
        }

        $limit = (int)$limit;
        if ($limit <= 0) $limit = 50;

        $this->_ensure_queue_table_exists();

        $now = date('Y-m-d H:i:s');
        $this->db
            ->where("(next_attempt IS NULL OR next_attempt <= ", null, false)
            ->where('status', 'pending');

        // Simpler select using query to allow next_attempt condition
        $q = $this->db->query("SELECT * FROM tec_agt_queue WHERE status = 'pending' AND (next_attempt IS NULL OR next_attempt <= ?) ORDER BY created_at ASC LIMIT ?", array($now, $limit));
        $rows = $q->result();

        $results = array();
        foreach ($rows as $r) {
            $results[] = $this->_process_queue_row($r);
        }

        if (is_cli()) {
            echo json_encode($results, JSON_PRETTY_PRINT|JSON_UNESCAPED_UNICODE) . "\n";
            return;
        }
        return $this->_json(array('ok'=>true,'processed'=>count($results),'results'=>$results));
    }

    private function _process_queue_row($row)
    {
        $id = (int)$row->id;
        $sale_id = (int)$row->sale_id;
        // Recarrega venda para verificar estado atual
        $sale = $this->db->get_where('tec_sales', array('id'=>$sale_id), 1)->row();

        // Se a venda já estiver VALIDA, não precisa de processamento
        if ($sale && isset($sale->agt_status) && strtoupper(trim($sale->agt_status)) === 'VALIDA') {
            $this->db->where('id',$id)->update('tec_agt_queue', array('status'=>'done','updated_at'=>date('Y-m-d H:i:s')));
            return array('id'=>$id,'sale_id'=>$sale_id,'status'=>'done','note'=>'already_valid');
        }

        // marca como processing
        $this->db->where('id',$id)->update('tec_agt_queue', array('status'=>'processing','updated_at'=>date('Y-m-d H:i:s')));

        $resp = array('ok'=>false,'error'=>'no_action');

        // Se já temos requestID, consultamos o estado em vez de reenviar a factura
        $requestID = ($sale && !empty($sale->agt_request_id)) ? trim((string)$sale->agt_request_id) : '';
        if ($requestID !== '') {
            $cfg = $this->_agt_cfg();
            if (!$cfg['ok']) {
                $resp = array('ok'=>false,'error'=>'cfg_error','details'=>$cfg);
            } else {
                $consulta = $this->_consultar_estado_agora($sale_id, $requestID, $cfg, 0);
                $resp = $consulta;
            }

            // Recarrega venda para verificar se ficou com request/status atualizado
            $sale = $this->db->get_where('tec_sales', array('id'=>$sale_id), 1)->row();
            if ($sale && isset($sale->agt_status) && strtoupper(trim($sale->agt_status)) === 'VALIDA') {
                $this->db->where('id',$id)->update('tec_agt_queue', array('status'=>'done','updated_at'=>date('Y-m-d H:i:s')));
                return array('id'=>$id,'sale_id'=>$sale_id,'status'=>'done','note'=>'validated');
            }
        } else {
            // Sem requestID: tenta reenviar a factura (registar_factura)
            try {
                $_GET['send'] = 1;
                $_GET['debug'] = 0;
                $resp = $this->registar_factura($sale_id);
            } catch (Exception $e) {
                $resp = array('ok'=>false,'error'=>$e->getMessage());
            }

            // Recarrega venda para ver se obteve requestID
            $sale = $this->db->get_where('tec_sales', array('id'=>$sale_id), 1)->row();
            if ($sale && !empty($sale->agt_request_id)) {
                $this->db->where('id',$id)->update('tec_agt_queue', array('status'=>'done','updated_at'=>date('Y-m-d H:i:s')));
                return array('id'=>$id,'sale_id'=>$sale_id,'status'=>'done','note'=>'registered');
            }
        }

        // Se chegamos aqui, houve falha — incrementa attempts e agenda próxima tentativa com backoff
        $attempts = (int)$row->attempts + 1;
        $backoffs = array(60,300,1800,7200,14400); // segundos: 1m,5m,30m,2h,4h
        $delay = isset($backoffs[$attempts-1]) ? $backoffs[$attempts-1] : 86400; // depois 1 dia
        $next = date('Y-m-d H:i:s', time() + $delay);
        $last_error = is_array($resp) ? json_encode($resp, JSON_UNESCAPED_UNICODE) : (string)$resp;

        $update = array('attempts'=>$attempts,'status'=>'pending','next_attempt'=>$next,'last_error'=>$last_error,'updated_at'=>date('Y-m-d H:i:s'));
        if ($attempts >= 10) {
            $update['status'] = 'failed';
        }
        $this->db->where('id',$id)->update('tec_agt_queue', $update);

        return array('id'=>$id,'sale_id'=>$sale_id,'status'=>$update['status'],'attempts'=>$attempts,'next_attempt'=>$next,'last_error'=>$last_error);
    }



    /* ===================== JSON HELPERS ===================== */

    private function _safe_json($raw)
    {
        if (!is_string($raw)) return array();
        $j = json_decode($raw, true);
        return is_array($j) ? $j : array();
    }

    private function _json($arr)
    {
        $this->output
            ->set_content_type('application/json; charset=utf-8')
            ->set_output(json_encode($arr));
        return;
    }

    /* =========================================================
     * 3) LISTAR FACTURAS - /fe/v1/listarFacturas
     * URL: /Agt/listar_facturas?send=1&from=YYYY-MM-DD&to=YYYY-MM-DD&size=50&debug=1
     * =======================================================*/
    public function listar_facturas()
    {
        $send  = (int)$this->input->get('send');
        $debug = (int)$this->input->get('debug');

        $from_ui = trim((string)$this->input->get('from'));
        $to_ui   = trim((string)$this->input->get('to'));
        $size    = (int)$this->input->get('size');
        if ($size <= 0) $size = 50;

        $from = $this->_norm_date_ymd($from_ui);
        $to   = $this->_norm_date_ymd($to_ui);

        if (!$from) $from = date('Y-m-d');
        if (!$to)   $to   = date('Y-m-d');

        if (!$send) {
            return $this->_json(array(
                'ok' => true,
                'http_code' => 200,
                'payload' => null,
                'debug_note' => 'Sem envio. Usa ?send=1&from=YYYY-MM-DD&to=YYYY-MM-DD&size=50'
            ));
        }

        $cfg = $this->_agt_cfg();
        if (!$cfg['ok']) return $this->_json($cfg);

        $payload = $this->_build_listar_facturas_rest_payload($cfg, $from, $to, $size);

        $url = rtrim($cfg['base_url'], '/') . '/fe/v1/listarFacturas';
        $res = $this->_http_json('POST', $url, $payload, $cfg['username'], $cfg['password'], $debug);

        $json = $this->_safe_json($res['body']);

        return $this->_json(array(
            'ok' => (bool)$res['ok'],
            'http_code' => (int)$res['http_code'],
            'payload' => $payload,
            'agt' => array(
                'ok' => (bool)$res['ok'],
                'http_code' => (int)$res['http_code'],
                'body' => $res['body'],
                'json' => $json,
                'error' => (string)$res['error'],
            ),
            'debug' => isset($res['_debug']) ? $res['_debug'] : null
        ));
    }

    private function _build_listar_facturas_rest_payload($cfg, $from, $to, $size = 50)
    {
        $submissionGUID = $this->_uuid_v4();
        $submissionTimeStamp = gmdate('Y-m-d\TH:i:s\Z');

        $softwareInfoDetail = array(
            'productId' => (string)$cfg['productId'],
            'productVersion' => (string)$cfg['productVersion'],
            'softwareValidationNumber' => (string)$cfg['softwareValidationNumber'],
        );

        // assinatura do software/produtor
        $jwsSoftwareSignature = $this->_jws_sign_json_with_filekey(
            $softwareInfoDetail,
            $cfg['software_private_key_path']
        );

        // assinatura do pedido (cliente)
        $forSign = array(
            'taxRegistrationNumber' => (string)$cfg['taxRegistrationNumber'],
            'queryStartDate'        => (string)$from,
            'queryEndDate'          => (string)$to,
            'size'                  => (int)$size,
        );

        $jwsSignature = $this->_jws_sign_json_with_filekey($forSign, $cfg['client_private_key_path']);

        return array(
            'schemaVersion' => '1.0',
            'submissionGUID' => $submissionGUID,
            'submissionTimeStamp' => $submissionTimeStamp,
            'softwareInfo' => array(
                'softwareInfoDetail' => $softwareInfoDetail,
                'jwsSoftwareSignature' => $jwsSoftwareSignature,
            ),
            'jwsSignature' => $jwsSignature,
            'taxRegistrationNumber' => (string)$cfg['taxRegistrationNumber'],
            'queryStartDate' => (string)$from,
            'queryEndDate' => (string)$to,
            'size' => (int)$size
        );
    }

    /* aceita "14/12/2025" ou "2025-12-14" */
    private function _norm_date_ymd($s)
    {
        $s = trim((string)$s);
        if ($s === '') return null;

        if (preg_match('/^\d{4}-\d{2}-\d{2}$/', $s)) return $s;

        if (preg_match('/^(\d{2})\/(\d{2})\/(\d{4})$/', $s, $m)) {
            return $m[3] . '-' . $m[2] . '-' . $m[1];
        }
        return null;
    }

    /* ===================== QUEUE ADMIN PANEL ===================== */

    // Painel de monitorização da fila
    public function queue_list()
    {
        // Apenas admin
        if (!$this->session->userdata('user_id')) {
            show_404();
        }

        $this->_ensure_queue_table_exists();

        // Estatísticas
        $stats = array(
            'pending' => (int)$this->db->where('status', 'pending')->count_all_results('tec_agt_queue'),
            'processing' => (int)$this->db->where('status', 'processing')->count_all_results('tec_agt_queue'),
            'done' => (int)$this->db->where('status', 'done')->count_all_results('tec_agt_queue'),
            'failed' => (int)$this->db->where('status', 'failed')->count_all_results('tec_agt_queue'),
        );

        // Lista (últimos 100)
        $rows = $this->db
            ->order_by('updated_at', 'DESC')
            ->limit(100)
            ->get('tec_agt_queue')
            ->result();

        // Enriquece com dados da venda
        foreach ($rows as $r) {
            $sale = $this->db->select('id, InvoiceNo, InvoiceType, grand_total, agt_status, agt_request_id')
                ->where('id', $r->sale_id)
                ->get('tec_sales')
                ->row();
            $r->sale = $sale;
        }

        $data = array(
            'stats' => $stats,
            'rows' => $rows,
            'page_title' => 'AGT Queue Monitor'
        );

        $this->load->view('agt_queue_panel', $data);
    }

    // API: retorna JSON com status da fila
    public function queue_status()
    {
        if (!$this->session->userdata('user_id')) {
            return $this->_json(array('ok'=>false,'error'=>'Unauthorized'));
        }

        $this->_ensure_queue_table_exists();

        $stats = array(
            'pending' => (int)$this->db->where('status', 'pending')->count_all_results('tec_agt_queue'),
            'processing' => (int)$this->db->where('status', 'processing')->count_all_results('tec_agt_queue'),
            'done' => (int)$this->db->where('status', 'done')->count_all_results('tec_agt_queue'),
            'failed' => (int)$this->db->where('status', 'failed')->count_all_results('tec_agt_queue'),
        );

        return $this->_json(array('ok'=>true,'stats'=>$stats));
    }

    // Reprocessar um item específico
    public function queue_reprocess($id = null)
    {
        if (!$this->session->userdata('user_id')) {
            show_404();
        }

        $id = (int)$id;
        if ($id <= 0) {
            $this->session->set_flashdata('error', 'ID inválido');
            redirect('agt/queue_list');
        }

        $this->_ensure_queue_table_exists();

        $row = $this->db->get_where('tec_agt_queue', array('id'=>$id), 1)->row();
        if (!$row) {
            $this->session->set_flashdata('error', 'Item não encontrado');
            redirect('agt/queue_list');
        }

        $sale_id = (int)$row->sale_id;

        // Processa
        $result = $this->_process_queue_row($row);

        $this->session->set_flashdata('message', 'Item processado: ' . json_encode($result));
        redirect('agt/queue_list');
    }

    // Remover item da fila
    public function queue_clear($id = null)
    {
        if (!$this->session->userdata('user_id')) {
            show_404();
        }

        $id = (int)$id;
        if ($id <= 0) {
            $this->session->set_flashdata('error', 'ID inválido');
            redirect('agt/queue_list');
        }

        $this->_ensure_queue_table_exists();

        $this->db->where('id', $id)->delete('tec_agt_queue');

        $this->session->set_flashdata('message', 'Item removido da fila');
        redirect('agt/queue_list');
    }

    /* =========================================================
     * LISTAR SÉRIES - /fe/v1/listarSeries
     * Query params: ?send=1&seriesCode=...&seriesYear=...&seriesStatus=...&documentType=...&establishmentNumber=...
     * Endpoint público: /agt/listar_series
     * =======================================================*/
    public function listar_series()
    {
        $send  = (int)$this->input->get('send');
        $debug = (int)$this->input->get('debug');

        // filtros opcionais (conforme doc)
        $seriesCode = trim((string)$this->input->get('seriesCode'));
        $seriesYear = trim((string)$this->input->get('seriesYear'));
        $seriesStatus = trim((string)$this->input->get('seriesStatus'));
        $documentType = trim((string)$this->input->get('documentType'));
        $establishmentNumber = trim((string)$this->input->get('establishmentNumber'));

        if (!$send) {
            return $this->_json(array(
                'ok'=>true,
                'http_code'=>200,
                'payload'=>null,
                'debug_note'=>'Sem envio. Usa ?send=1&seriesCode=...&seriesYear=...&seriesStatus=...&documentType=...&establishmentNumber=10'
            ));
        }

        $cfg = $this->_agt_cfg();
        if (!$cfg['ok']) return $this->_json($cfg);

        $payload = $this->_build_listar_series_payload(
            $cfg,
            $seriesCode,
            $seriesYear,
            $seriesStatus,
            $documentType,
            $establishmentNumber
        );

        $url = rtrim($cfg['base_url'],'/') . '/fe/v1/listarSeries';
        $res = $this->_http_json('POST', $url, $payload, $cfg['username'], $cfg['password'], $debug);

        $json = $this->_safe_json($res['body']);

        $uniqueByType = array();
        $wanted = array('FT','GF','FR','NC','RC');

        if (isset($json['seriesInfo']) && is_array($json['seriesInfo'])) {

            // ordenar: A primeiro, depois ano maior, depois id maior
            usort($json['seriesInfo'], function($a, $b){

                $sa = isset($a['seriesStatus']) ? (string)$a['seriesStatus'] : '';
                $sb = isset($b['seriesStatus']) ? (string)$b['seriesStatus'] : '';

                $pa = ($sa === 'A') ? 2 : (($sa === 'U') ? 1 : 0);
                $pb = ($sb === 'A') ? 2 : (($sb === 'U') ? 1 : 0);

                if ($pa > $pb) return -1;
                if ($pa < $pb) return 1;

                $ya = isset($a['seriesYear']) ? (int)$a['seriesYear'] : 0;
                $yb = isset($b['seriesYear']) ? (int)$b['seriesYear'] : 0;

                if ($ya > $yb) return -1;
                if ($ya < $yb) return 1;

                $ia = isset($a['id']) ? (int)$a['id'] : 0;
                $ib = isset($b['id']) ? (int)$b['id'] : 0;

                if ($ia > $ib) return -1;
                if ($ia < $ib) return 1;

                return 0;
            });

            foreach ($json['seriesInfo'] as $s) {
                $type = isset($s['documentType']) ? strtoupper(trim((string)$s['documentType'])) : '';
                if ($type === '') continue;

                if (!in_array($type, $wanted, true)) continue;

                if (!isset($uniqueByType[$type])) {
                    $uniqueByType[$type] = array(
                        'id' => isset($s['id']) ? $s['id'] : null,
                        'documentType' => $type,
                        'seriesCode' => isset($s['seriesCode']) ? $s['seriesCode'] : null,
                        'seriesYear' => isset($s['seriesYear']) ? $s['seriesYear'] : null,
                        'seriesStatus' => isset($s['seriesStatus']) ? $s['seriesStatus'] : null,
                        'nextNumber' => isset($s['nextNumber']) ? $s['nextNumber'] : null,
                        'establishmentNumber' => isset($s['establishmentNumber']) ? $s['establishmentNumber'] : null,
                    );
                }

                if (count($uniqueByType) === count($wanted)) break;
            }
        }

        return $this->_json(array(
            'ok' => $res['ok'],
            'http_code' => (int)$res['http_code'],
            'payload' => $payload,
            'uniqueByType' => $uniqueByType,
            'agt' => array(
                'ok' => $res['ok'],
                'http_code' => (int)$res['http_code'],
                'body' => $res['body'],
                'json' => $json,
                'error' => (string)$res['error'],
            ),
            'debug' => isset($res['_debug']) ? $res['_debug'] : null
        ));
    }

    private function _build_listar_series_payload($cfg, $seriesCode, $seriesYear, $seriesStatus, $documentType, $establishmentNumber)
    {
        $submissionTimeStamp = gmdate('Y-m-d\TH:i:s\Z');

        $softwareInfoDetail = array(
            'productId' => (string)$cfg['productId'],
            'productVersion' => (string)$cfg['productVersion'],
            'softwareValidationNumber' => (string)$cfg['softwareValidationNumber'],
        );

        // Assina softwareInfo usando a chave software já gravada em ficheiro
        $jwsSoftwareSignature = '';
        if (!empty($cfg['software_private_key_path'])) {
            $jwsSoftwareSignature = $this->_jws_sign_json_with_filekey($softwareInfoDetail, $cfg['software_private_key_path']);
        }

        $forSign = array(
            'taxRegistrationNumber' => (string)$cfg['taxRegistrationNumber'],
        );

        if ($seriesCode !== '') $forSign['seriesCode'] = (string)$seriesCode;
        if ($seriesYear !== '') $forSign['seriesYear'] = (string)$seriesYear;
        if ($seriesStatus !== '') $forSign['seriesStatus'] = (string)$seriesStatus;
        if ($documentType !== '') $forSign['documentType'] = (string)$documentType;
        if ($establishmentNumber !== '') $forSign['establishmentNumber'] = (string)$establishmentNumber;

        $jwsSignature = $this->_jws_sign_json_with_filekey($forSign, $cfg['client_private_key_path']);

        $payload = array(
            'schemaVersion' => '1.0',
            'taxRegistrationNumber' => (string)$cfg['taxRegistrationNumber'],
            'submissionTimeStamp' => $submissionTimeStamp,
            'jwsSignature' => $jwsSignature,
            'softwareInfo' => array(
                'softwareInfoDetail' => $softwareInfoDetail,
                'jwsSoftwareSignature' => $jwsSoftwareSignature,
            ),
        );

        if ($seriesCode !== '') $payload['seriesCode'] = (string)$seriesCode;
        if ($seriesYear !== '') $payload['seriesYear'] = (string)$seriesYear;
        if ($seriesStatus !== '') $payload['seriesStatus'] = (string)$seriesStatus;
        if ($documentType !== '') $payload['documentType'] = (string)$documentType;
        if ($establishmentNumber !== '') $payload['establishmentNumber'] = (string)$establishmentNumber;

        return $payload;
    }

    /* =========================================================
     * SOLICITAR SÉRIE - /fe/v1/solicitarSerie
     * Endpoint para registar séries na AGT
     * Query params: ?send=1&documentType=FT&seriesYear=2026&establishmentNumber=1
     * =======================================================*/
    public function solicitar_serie()
    {
        $send  = (int)$this->input->get('send');
        $debug = (int)$this->input->get('debug');

        $documentType = trim((string)$this->input->get('documentType'));
        $seriesYear = trim((string)$this->input->get('seriesYear'));
        $establishmentNumber = trim((string)$this->input->get('establishmentNumber'));

        if (!$send) {
            return $this->_json(array(
                'ok'=>true,
                'http_code'=>200,
                'message'=>'Sem envio. Use ?send=1&documentType=FT&seriesYear=2026&establishmentNumber=1'
            ));
        }

        if ($documentType === '') {
            return $this->_json(array('ok'=>false, 'error'=>'documentType é obrigatório (FT, FR, GF, NC, RC)'));
        }

        $cfg = $this->_agt_cfg();
        if (!$cfg['ok']) return $this->_json($cfg);

        if ($seriesYear === '') {
            $seriesYear = date('Y'); // ano actual
        }

        if ($establishmentNumber === '') {
            $establishmentNumber = '1'; // padrão
        }

        $payload = $this->_build_solicitar_serie_payload($cfg, $documentType, $seriesYear, $establishmentNumber);

        $url = rtrim($cfg['base_url'],'/') . '/fe/v1/solicitarSerie';
        $res = $this->_http_json('POST', $url, $payload, $cfg['username'], $cfg['password'], $debug);

        $json = $this->_safe_json($res['body']);

        return $this->_json(array(
            'ok' => $res['ok'],
            'http_code' => (int)$res['http_code'],
            'payload' => $payload,
            'response' => $json,
            'agt' => array(
                'ok' => $res['ok'],
                'http_code' => (int)$res['http_code'],
                'body' => $res['body'],
                'json' => $json,
                'error' => (string)$res['error'],
            ),
            'debug' => isset($res['_debug']) ? $res['_debug'] : null
        ));
    }

    private function _build_solicitar_serie_payload($cfg, $documentType, $seriesYear, $establishmentNumber)
    {
        $submissionUUID = $this->_generate_uuid();
        $submissionTimeStamp = gmdate('Y-m-d\TH:i:s\Z');

        $softwareInfoDetail = array(
            'productId' => (string)$cfg['productId'],
            'productVersion' => (string)$cfg['productVersion'],
            'softwareValidationNumber' => (string)$cfg['softwareValidationNumber'],
        );

        // Assina softwareInfo
        $jwsSoftwareSignature = '';
        if (!empty($cfg['software_private_key_path'])) {
            $jwsSoftwareSignature = $this->_jws_sign_json_with_filekey($softwareInfoDetail, $cfg['software_private_key_path']);
        }

        $forSign = array(
            'taxRegistrationNumber' => (string)$cfg['taxRegistrationNumber'],
            'seriesYear' => (string)$seriesYear,
            'documentType' => (string)$documentType,
            'establishmentNumber' => (string)$establishmentNumber, // Aceita "SEDE" ou número
            'seriesContingencyIndicator' => 'N'
        );

        $jwsSignature = $this->_jws_sign_json_with_filekey($forSign, $cfg['client_private_key_path']);

        $payload = array(
            'schemaVersion' => '1.2',
            'submissionUUID' => $submissionUUID,
            'taxRegistrationNumber' => (string)$cfg['taxRegistrationNumber'],
            'submissionTimeStamp' => $submissionTimeStamp,
            'softwareInfo' => array(
                'softwareInfoDetail' => $softwareInfoDetail,
                'jwsSoftwareSignature' => $jwsSoftwareSignature,
            ),
            'seriesYear' => (string)$seriesYear,
            'documentType' => (string)$documentType,
            'establishmentNumber' => (string)$establishmentNumber,
            'jwsSignature' => $jwsSignature,
            'seriesContingencyIndicator' => 'N'
        );

        return $payload;
    }

    /* =========================================================
     * CRIAR TODAS AS SÉRIES AUTOMATICAMENTE
     * Cria séries para FT, FR, GF, NC, RC de uma só vez
     * =======================================================*/
    public function criar_todas_series()
    {
        $send = (int)$this->input->get('send');
        $seriesYear = trim((string)$this->input->get('seriesYear'));
        $establishmentNumber = trim((string)$this->input->get('establishmentNumber'));

        if (!$send) {
            return $this->_json(array(
                'ok'=>true,
                'message'=>'Use ?send=1&seriesYear=2026&establishmentNumber=1'
            ));
        }

        if ($seriesYear === '') {
            $seriesYear = date('Y');
        }

        if ($establishmentNumber === '') {
            $establishmentNumber = '1';
        }

        $cfg = $this->_agt_cfg();
        if (!$cfg['ok']) {
            return $this->_json(array('ok'=>false, 'error'=>'Erro na configuração AGT', 'cfg'=>$cfg));
        }

        $tipos = array('FT', 'FR', 'GF', 'NC', 'RC');
        $resultados = array();
        $sucessos = 0;
        $erros = 0;

        foreach ($tipos as $tipo) {
            // Chamar directamente a lógica de criação
            $payload = $this->_build_solicitar_serie_payload($cfg, $tipo, $seriesYear, $establishmentNumber);
            $url = rtrim($cfg['base_url'],'/') . '/fe/v1/solicitarSerie';
            $res = $this->_http_json('POST', $url, $payload, $cfg['username'], $cfg['password'], 0);
            $json = $this->_safe_json($res['body']);

            $resultado = array(
                'tipo' => $tipo,
                'ok' => $res['ok'],
                'http_code' => (int)$res['http_code'],
                'response' => $json,
                'error' => (string)$res['error']
            );

            $resultados[$tipo] = $resultado;
            
            if ($res['ok'] && $res['http_code'] == 200) {
                $sucessos++;
            } else {
                $erros++;
            }
            
            // Pequeno delay entre requests
            usleep(500000); // 0.5 segundos
        }

        return $this->_json(array(
            'ok' => ($erros === 0),
            'message' => "Criadas {$sucessos} séries com sucesso, {$erros} erros",
            'sucessos' => $sucessos,
            'erros' => $erros,
            'resultados' => $resultados
        ));
    }

    private function _generate_uuid()
    {
        return sprintf(
            '%04x%04x-%04x-%04x-%04x-%04x%04x%04x',
            mt_rand(0, 0xffff), mt_rand(0, 0xffff),
            mt_rand(0, 0xffff),
            mt_rand(0, 0x0fff) | 0x4000,
            mt_rand(0, 0x3fff) | 0x8000,
            mt_rand(0, 0xffff), mt_rand(0, 0xffff), mt_rand(0, 0xffff)
        );
    }

    /**
     * Converte uma Factura Proforma (FP) em Factura definitiva (FT)
     * Cria uma nova venda duplicando todos os dados da FP mas com InvoiceType=FT e novo número
     */
    private function _convert_fp_to_ft($fp_sale_id)
    {
        $fp_sale_id = (int)$fp_sale_id;
        
        // 1. Carrega a FP original
        $fp_sale = $this->db->get_where('tec_sales', array('id'=>$fp_sale_id), 1)->row_array();
        if (!$fp_sale) {
            return array('ok'=>false, 'error'=>'FP não encontrada');
        }

        // 2. Carrega itens da FP
        $fp_items = $this->db->get_where('tec_sale_items', array('sale_id'=>$fp_sale_id))->result_array();

        // 3. Gera novo número FT (seguindo lógica AGT)
        $invoice_type = 'FT';
        $invoice_year = date("Y");

        $last_num_row = $this->db->select('id, number, agt_series_code')
            ->from('tec_numbering')
            ->where('InvoiceYear', $invoice_year)
            ->where('InvoiceType', $invoice_type)
            ->order_by('id', 'DESC')
            ->limit(1)
            ->get()
            ->row_array();

        $invoice_number = 1;
        $agt_series_code = '';

        if (!empty($last_num_row)) {
            $invoice_number = ((int)$last_num_row['number']) + 1;
            $agt_series_code = isset($last_num_row['agt_series_code']) ? trim((string)$last_num_row['agt_series_code']) : '';
        }

        if ($agt_series_code === '') {
            $col = $invoice_type . '-seriesCode';
            $query = $this->db->select("`{$col}` AS sc", false)
                ->from('tec_settings')
                ->limit(1)
                ->get();
            $set = ($query && $query->num_rows() > 0) ? $query->row_array() : array();
            $agt_series_code = isset($set['sc']) ? trim((string)$set['sc']) : '';
        }

        if ($agt_series_code === '') {
            return array('ok'=>false, 'error'=>'Sem série FT configurada em tec_settings');
        }

        $invoice_no = $invoice_type . " " . $agt_series_code . "/" . $invoice_number;

        // 4. Gera novo hash
        $date = date('Y-m-d H:i:s');
        
        // Busca hash anterior (última FT ou da própria FP se não houver)
        $before_hash = '';
        $last_sale = $this->db->select('Hash')
            ->from('tec_sales')
            ->where('id <', $fp_sale_id)
            ->order_by('id', 'DESC')
            ->limit(1)
            ->get()
            ->row();
        if ($last_sale && isset($last_sale->Hash)) {
            $before_hash = substr($last_sale->Hash, 0, 4);
        }

        $hash_str = date("Y-m-d", strtotime($date)) . ";"
            . date("Y-m-d", strtotime($date)) . "T" . date("H:i:s", strtotime($date)) . ";"
            . $invoice_no . ";"
            . str_replace(",", "", number_format($fp_sale['grand_total'], 2)) . ";"
            . $before_hash;

        $hash_no = $hash_str;

        $sign_key_row = $this->db->get_where('tec_signkey', array('id'=>1), 1)->row_array();
        if (!$sign_key_row || !isset($sign_key_row['private'])) {
            return array('ok'=>false, 'error'=>'Chave privada não encontrada');
        }

        $privateKey = $sign_key_row['private'];
        openssl_sign($hash_str, $crypttext, $privateKey, "sha1");
        $hash = base64_encode($crypttext);

        // 5. Cria nova venda FT (duplica dados da FP)
        $new_sale = $fp_sale;
        unset($new_sale['id']); // Remove ID para criar novo
        
        $new_sale['InvoiceType'] = 'FT';
        $new_sale['InvoiceNo'] = $invoice_no;
        $new_sale['InvoiceYear'] = $invoice_year;
        $new_sale['Hash'] = $hash;
        $new_sale['hash_no'] = $hash_no;
        $new_sale['HashControl'] = '1';
        $new_sale['date'] = $date;
        $new_sale['InvoiceStatusDate'] = date("Y-m-d", strtotime($date)) . "T" . date("H:i:s", strtotime($date));
        $new_sale['OriginatingON'] = $fp_sale['InvoiceNo']; // Referencia a FP original
        $new_sale['mesa'] = 'N';
        $new_sale['status'] = 'no paid';
        $new_sale['paid'] = 0;
        
        // Limpa campos AGT (nova factura não tem estado AGT ainda)
        $new_sale['agt_status'] = null;
        $new_sale['agt_request_id'] = null;
        $new_sale['agt_submitted_at'] = null;
        $new_sale['agt_error_json'] = null;

        $this->db->insert('tec_sales', $new_sale);
        $new_sale_id = $this->db->insert_id();

        if (!$new_sale_id) {
            return array('ok'=>false, 'error'=>'Erro ao criar nova FT');
        }

        // 6. Duplica itens
        foreach ($fp_items as $item) {
            $new_item = $item;
            unset($new_item['id']);
            $new_item['sale_id'] = $new_sale_id;
            $this->db->insert('tec_sale_items', $new_item);
        }

        // 7. Actualiza numeração
        if (!empty($last_num_row['id'])) {
            $this->db->where('id', $last_num_row['id']);
            $this->db->update('tec_numbering', array('number'=>$invoice_number));
        } else {
            $this->db->insert('tec_numbering', array(
                'InvoiceType' => 'FT',
                'InvoiceYear' => $invoice_year,
                'number' => $invoice_number,
                'agt_series_code' => $agt_series_code
            ));
        }

        // 8. Marca a FP original como convertida (opcional)
        $this->db->where('id', $fp_sale_id);
        $this->db->update('tec_sales', array(
            'mesa' => 'CONVERTIDA',
            'note' => (isset($fp_sale['note']) ? $fp_sale['note'] . ' | ' : '') . 'Convertida para FT: ' . $invoice_no
        ));

        return array(
            'ok' => true,
            'new_sale_id' => $new_sale_id,
            'new_invoice_no' => $invoice_no,
            'original_fp_id' => $fp_sale_id
        );
    }
}
