<?php defined('BASEPATH') OR exit('No direct script access allowed');

class SaftValidator extends CI_Controller {

  public function index() {
    header('Content-Type: application/json; charset=utf-8');

    // 1) Resolve caminho do XML
    $req = $this->input->get('xml', true);
    $xmlPath = $this->resolveXmlPath($req);        // aceita vazio, só o nome, ou caminho parcial
    $xsdPath = FCPATH.'uploads/XSD.xml';

    if (!$xmlPath || !file_exists($xmlPath)) {
      echo json_encode(['ok'=>false,'errors'=>['XML não encontrado: '.$req]], JSON_PRETTY_PRINT|JSON_UNESCAPED_UNICODE);
      return;
    }
    if (!file_exists($xsdPath)) {
      echo json_encode(['ok'=>false,'errors'=>['XSD não encontrado em '.$xsdPath]], JSON_PRETTY_PRINT|JSON_UNESCAPED_UNICODE);
      return;
    }

    // 2) Valida: XSD + regras de negócio
    $out = $this->validate($xmlPath, $xsdPath);
    echo json_encode($out, JSON_PRETTY_PRINT|JSON_UNESCAPED_UNICODE);
  }

  private function resolveXmlPath($input) {
    // Sem parâmetro -> valida o último ficheiro em uploads/xml
    $dir = rtrim(FCPATH, '/\\').DIRECTORY_SEPARATOR.'uploads'.DIRECTORY_SEPARATOR.'xml'.DIRECTORY_SEPARATOR;

    if (!$input) {
      $files = glob($dir.'*.xml');
      rsort($files);
      return $files ? $files[0] : null;
    }

    // Se veio só o nome (ex.: saft.xml)
    if (basename($input) === $input) {
      return $dir.$input;
    }

    // Se veio com caminho começado por /uploads/... (ou \uploads\... no Windows)
    $in = str_replace(['\\','//'], '/', $input);
    if (strpos($in, '/uploads/') === 0) {
      return rtrim(FCPATH, '/\\').$in;
    }

    // Caso tenha passado um caminho absoluto do SO
    if (file_exists($input)) return $input;

    // Fallback: tenta como relativo a uploads/xml
    return $dir.basename($input);
  }

  private function validate($xmlPath, $xsdPath) {
    libxml_use_internal_errors(true);
    $doc = new DOMDocument();
    $doc->load($xmlPath);
    $schemaOk = $doc->schemaValidate($xsdPath);

    $errors = [];
    if (!$schemaOk) {
      foreach (libxml_get_errors() as $e) $errors[] = trim($e->message).' @ line '.$e->line;
    }
    libxml_clear_errors();

    // Regras de negócio
    $xp = new DOMXPath($doc);
    $xp->registerNamespace('ns','urn:OECD:StandardAuditFile-Tax:AO_1.01_01');

    // 1) Isenção requer motivo
    foreach ($xp->query('//ns:SourceDocuments//ns:Line') as $line) {
      $pct  = $xp->evaluate('number(.//ns:TaxPercentage)', $line);
      $code = $xp->evaluate('string(.//ns:TaxCode)', $line);
      if ($pct == 0 || in_array($code, ['ISE','NS','OUT'])) {
        $hasCode   = $xp->evaluate('string(.//ns:TaxExemptionCode)', $line) !== '';
        $hasReason = $xp->evaluate('string(.//ns:TaxExemptionReason)', $line) !== '';
        if (!($hasCode && $hasReason)) {
          $ln = $xp->evaluate('string(ns:LineNumber)', $line);
          $errors[] = "Linha {$ln}: falta TaxExemptionCode/Reason quando TaxPercentage=0 ou TaxCode ∈ (ISE/NS/OUT).";
        }
      }
    }

    // 2) TaxTable apenas NOR/ISE/OUT/NS
    if ($xp->query('//ns:TaxTable/ns:TaxTableEntry[not(ns:TaxCode="NOR" or ns:TaxCode="ISE" or ns:TaxCode="OUT" or ns:TaxCode="NS")]')->length > 0) {
      $errors[] = 'TaxTable contém códigos fora do padrão (NOR/ISE/OUT/NS).';
    }

    // 3) NOR tem de ser 14%
    if ($xp->query('//ns:TaxTable/ns:TaxTableEntry[ns:TaxCode="NOR" and ns:TaxPercentage!=14]')->length > 0) {
      $errors[] = 'TaxTable: NOR deve ter TaxPercentage=14.';
    }

    return ['ok'=>count($errors)===0, 'errors'=>$errors];
  }
}
