<?php
defined('BASEPATH') OR exit('No direct script access allowed');

class Api extends MY_Controller {

    // Defaults (will be overridden from DB `tec_settings` when available)
    private $minfin_user = 'ws.hml.sifpos';
    private $minfin_pass = 'mfn697282025';
    private $minfin_base = 'https://sifp.minfin.gov.ao/sigt/contribuinte/v5/obter?';

    function __construct() {
        parent::__construct();
        header("Access-Control-Allow-Origin:*");
        header("Access-Control-Allow-Methods: GET, POST, OPTIONS, PUT, DELETE");
        header("Access-Control-Allow-Headers: X-API-KEY, Origin, X-Requested-With, Content-Type, Accept, Access-Control-Request-Method, Yippin_token,Basic");
        header('Content-type:application/json');

        $this->load->library('ion_auth');
        $this->load->library('form_validation');
        $this->load->helper('database_helper');

        $this->load->model('products_model');
        $this->load->model('categories_model');
        $this->load->model('gift_cards_model');
        $this->load->model('auth_model');
        $this->load->model('customers_model');
        $this->load->model('suppliers_model');
        $this->load->helper(array('url', 'form'));

        // Ensure DB is available and try to load AGT/MinFin settings from `tec_settings`.
        // Expected columns: `agt_username`, `agt_password`, `agt_base_url`.
        $this->load->database();
        try {
            $settings = $this->db->get('tec_settings')->row();
            if ($settings) {
                if (!empty($settings->agt_username)) {
                    $this->minfin_user = $settings->agt_username;
                }
                if (!empty($settings->agt_password)) {
                    $this->minfin_pass = $settings->agt_password;
                }
                if (!empty($settings->agt_base_url)) {
                    // Remove apenas barras finais; não usar rtrim with substring
                    $this->minfin_base = rtrim($settings->agt_base_url, '/');
                }
            }
        } catch (Exception $e) {
            // If settings table doesn't exist or DB error occurs, keep defaults.
        }
    }

    /* =========================================================
     *  LICENÇA / STATUS
     * =======================================================*/

    function register_logged_status() {
        $online_date       = date("Y-m-d H:i:s");
        $logged_status     = $this->input->post('logged_status');
        $logged_ip         = $this->input->post('logged_ip');
        $TaxRegistrationNumber = $this->input->post('TaxRegistrationNumber');
        $licence           = $this->input->post('licence');

        update_row(
            "tec_licences",
            array(
                "online_date"   => $online_date,
                "logged_status" => $logged_status,
                "logged_ip"     => $logged_ip
            ),
            array(
                "TaxRegistrationNumber" => $TaxRegistrationNumber,
                "licence"               => $licence
            )
        );

        echo 1;
    }

    /* =========================================================
     *  CATEGORIAS
     * =======================================================*/

    function categories() {
        $categories = get_rows_left_join(
            "tec_categories",
            array(),
            "tec_categories.name asc",
            array(),
            "tec_categories.*, tec_printers.title",
            array(),
            array(array("table" => "tec_printers", "where" => "tec_categories.printer_id=tec_printers.printer_id"))
        );

        $data = array(
            "success"    => true,
            "message"    => "Got Data",
            "categories" => $categories,
        );
        echo json_encode($data);
    }

    function get_category($id) {
        $category = $this->site->getCategoryByID($id);
        $data = array(
            "success"    => true,
            "message"    => "Got Data",
            "categories" => $category,
        );
        echo json_encode($data);
    }

    function save_category() {
        $id         = $this->input->post('id');
        $inser_data = array();
        $data       = array();

        $inser_data["name"]       = $this->input->post('name');
        $inser_data["code"]       = $this->input->post('code');
        $inser_data["printer_id"] = $this->input->post('printer_id');
        $file_type                = $this->input->post('file_type');

        if ($file_type) {
            $croppedImag = $this->input->post('croppedImag');
            $dataURL     = str_replace('data:image/jpeg;base64,', '', $croppedImag);
            $dataURL     = str_replace(' ', '+', $dataURL);
            $image       = base64_decode($dataURL);
            $filename    = date("d-m-Y-h-i-s") . '.jpeg';

            $path = set_realpath('uploads/thumbs/');
            file_put_contents($path . $filename, $image);
            $inser_data["image"] = $filename;
        }

        if ($id) {
            update_row("tec_categories", $inser_data, array("id" => $id));
            $data["success"] = true;
            $data["message"] = lang('category_updated');
        } else {
            create_row("tec_categories", $inser_data);
            $data["success"] = true;
            $data["message"] = lang('category_added');
        }
        echo json_encode($data);
    }

    function delete_category() {
        $id   = $this->input->post('id');
        $data = array();

        if ($this->categories_model->deleteCategory($id)) {
            $data["success"] = true;
            $data["message"] = lang("category_deleted");
        } else {
            $data["success"] = false;
            $data["message"] = lang("category_deleted");
        }
        echo json_encode($data);
    }

    function get_printers() {
        $prints = get_rows("tec_printers", array(), "title asc");
        $data   = array(
            "success" => true,
            "message" => "Got Data",
            "prints"  => $prints,
        );
        echo json_encode($data);
    }

    /* =========================================================
     *  PRODUTOS (DETALHE)
     * =======================================================*/

    function product($id) {
        $where             = array('tec_products.id' => $id);
        $product           = get_rows_join(
            "tec_products",
            $where,
            "",
            array(),
            "tec_products.id as pid, tec_products.image as image,category_id, tec_products.code as code,
             tec_products.name as pname, type, details, tec_categories.name as cname, quantity, tax, tax_method,
             cost, price, barcode_symbology",
            array(),
            array(array("table" => "tec_categories", "where" => "tec_products.category_id=tec_categories.id"))
        );
        $data              = array();

        if (count($product) > 0) {
            $data["success"]           = true;
            $data["message"]           = "Got Data";
            $product[0]["type_text"]   = lang($product[0]["type"]);
            $product[0]["price"]       = $this->tec->formatMoney($product[0]["price"]);
            $product[0]["cost"]        = $this->tec->formatMoney($product[0]["cost"]);
            $product[0]["tax_method"]  = $product[0]["tax_method"] == 0 ? lang('inclusive') : lang('exclusive');
            $product[0]["quantity"]    = $this->tec->formatMoney($product[0]["quantity"]);
            $combo_items               = $product[0]["type"] == 'combo'
                ? $this->products_model->getComboItemsByPID($product[0]["pid"])
                : NULL;

            $data["combo_items"] = $combo_items;
            $data["data"]        = $product[0];
        } else {
            $data["success"] = false;
            $data["message"] = "no Data";
        }
        echo json_encode($data);
    }

    /* =========================================================
     *  GIFT CARDS
     * =======================================================*/

    function gift_cards() {
        $gift_cards = get_rows_left_join(
            "tec_gift_cards",
            array(),
            "",
            array(),
            "tec_gift_cards.id as id, card_no, value, balance,
             tec_customers.name,
             CONCAT(tec_users.first_name, ' ', tec_users.last_name) as created_by",
            array(),
            array(
                array("table" => "tec_users", "where" => "tec_users.id=tec_gift_cards.created_by"),
                array("table" => "tec_customers", "where" => "tec_customers.id=tec_gift_cards.customer_id")
            )
        );
        $data       = array(
            "success"    => true,
            "gift_cards" => $gift_cards,
        );
        echo json_encode($data);
    }

    function get_gift_card($id) {
        $gift_card = $this->gift_cards_model->getGiftCardByID($id);
        $data      = array(
            "success"   => true,
            "message"   => "Got Data",
            "gift_card" => $gift_card,
        );
        echo json_encode($data);
    }

    function save_gift_card() {
        $id = $this->input->post('id');

        if ($id) {
            $gift_card   = $this->gift_cards_model->getGiftCardByID($id);
            $update_data = array(
                'card_no'    => $this->input->post('card_no'),
                'value'      => $this->input->post('value'),
                'balance'    => ($this->input->post('value') - $gift_card->value) + $gift_card->balance,
                'customer_id'=> $this->input->post('customer_id'),
                'expiry'     => $this->input->post('expiry'),
                'created_by' => $this->input->post('created_by'),
            );
            update_row("tec_gift_cards", $update_data, array("id" => $id));
            $data["success"] = true;
            $data["message"] = lang('gift_card_updated');
        } else {
            $inser_data = array(
                'card_no'    => $this->input->post('card_no'),
                'value'      => $this->input->post('value'),
                'balance'    => $this->input->post('value'),
                'customer_id'=> $this->input->post('customer_id'),
                'expiry'     => $this->input->post('expiry'),
                'created_by' => $this->input->post('created_by'),
            );
            create_row("tec_gift_cards", $inser_data);
            $data["success"] = true;
            $data["message"] = lang('gift_card_added');
        }
        echo json_encode($data);
    }

    function delete_gift_card() {
        $id = $this->input->post('id');
        $this->gift_cards_model->deleteGiftCard($id);
        $data["success"] = true;
        $data["message"] = lang('category_deleted');
        echo json_encode($data);
    }

    /* =========================================================
     *  GRUPOS / UTILIZADORES
     * =======================================================*/

    function groups() {
        $groups = $this->ion_auth->groups()->result_array();
        $data   = array(
            "success" => true,
            "groups"  => $groups,
        );
        echo json_encode($data);
    }

    function users() {
        $users = $this->site->getAllUsers();
        $data  = array(
            "success" => true,
            "users"   => $users,
        );
        echo json_encode($data);
    }

    function get_user($id) {
        $user            = $this->ion_auth->user($id)->row();
        $user->group_id  = base64_decode($user->group_id);
        $data            = array(
            "success" => true,
            "message" => "Got Data",
            "user"    => $user,
        );
        echo json_encode($data);
    }

    function save_user() {
        $id = $this->input->post('id');

        if ($id) {
            $update_data = array(
                'first_name'          => $this->input->post('first_name'),
                'last_name'           => $this->input->post('last_name'),
                'username'            => $this->input->post('username'),
                'email'               => $this->input->post('email'),
                'phone'               => $this->input->post('phone'),
                'gender'              => $this->input->post('gender'),
                'active'              => $this->input->post('active'),
                'group_id'            => $this->input->post('group_id') ? base64_encode($this->input->post('group_id')) : base64_encode('2'),
                'box_closure'         => $this->input->post('box_closure') ? $this->input->post('box_closure') : 0,
                'unti_value_discount' => $this->input->post('unti_value_discount') ? $this->input->post('unti_value_discount') : 0,
                'settings'            => $this->input->post('settings') ? $this->input->post('settings') : 0,
                'reports'             => $this->input->post('reports') ? $this->input->post('reports') : 0,
            );
            if ($this->input->post('password')) {
                $update_data['password'] = $this->input->post('password');
            }

            if ($this->ion_auth->update($id, $update_data)) {
                $data["success"] = true;
                $data["message"] = lang('user_updated');
            } else {
                $data["success"] = false;
                $data["message"] = "please try again";
            }
        } else {
            $username = strtolower($this->input->post('username'));
            $email    = strtolower($this->input->post('email'));
            $password = $this->input->post('password');

            $additional_data = array(
                'first_name'          => $this->input->post('first_name'),
                'last_name'           => $this->input->post('last_name'),
                'phone'               => $this->input->post('phone'),
                'gender'              => $this->input->post('gender'),
                'email'               => $this->input->post('email'),
                'box_closure'         => $this->input->post('box_closure') ? $this->input->post('box_closure') : 0,
                'unti_value_discount' => $this->input->post('unti_value_discount') ? $this->input->post('unti_value_discount') : 0,
                'settings'            => $this->input->post('settings') ? $this->input->post('settings') : 0,
                'reports'             => $this->input->post('reports') ? $this->input->post('reports') : 0,
                'group_id'            => $this->input->post('group_id') ? base64_encode($this->input->post('group_id')) : base64_encode('2'),
            );

            $notify = $this->input->post('notify');
            $active = $this->input->post('active');

            if ($this->ion_auth->register($username, $password, $email, $additional_data, $active, $notify)) {
                $data["success"] = true;
                $data["message"] = lang('gift_card_added');
            } else {
                $data["success"] = false;
                $data["message"] = "please try again";
            }
        }

        echo json_encode($data);
    }

    function delete_user() {
        $id = $this->input->post('id');

        if ($this->auth_model->delete_user($id)) {
            $data["success"] = true;
            $data["message"] = lang('user_deleted');
        } else {
            $data["success"] = false;
            $data["message"] = "please try again";
        }

        echo json_encode($data);
    }

    /* =========================================================
     *  CLIENTES
     * =======================================================*/

    function customers() {
        $customers = $this->site->getAllCustomers();
        $data      = array(
            "success"   => true,
            "customers" => $customers,
        );
        echo json_encode($data);
    }

    function get_customer($id) {
        $customer = $this->customers_model->getCustomerByID($id);
        $data     = array(
            "success"  => true,
            "message"  => "Got Data",
            "customer" => $customer,
        );
        echo json_encode($data);
    }

    function save_customer() {
        $id = $this->input->post('id');

        if ($id) {
            $update_data = array(
                'name'        => $this->input->post('name'),
                'email'       => $this->input->post('email'),
                'phone'       => $this->input->post('phone'),
                'cf1'         => $this->input->post('cf1'),
                'endereco'    => $this->input->post('bairro'),
                'numero'      => $this->input->post('numero'),
                'bairro'      => $this->input->post('bairro'),
                'cidade'      => $this->input->post('cidade'),
                'estado'      => $this->input->post('estado'),
                'data'        => date("Y-m-d"),
                'AccountID'   => 0,
                'Country'     => $this->input->post('Country'),
                'complemento' => $this->input->post('bairro'),
                'cep'         => $this->input->post('cep'),
            );

            if ($this->customers_model->updateCustomer($id, $update_data)) {
                $data["success"] = true;
                $data["message"] = lang('customer_updated');
            } else {
                $data["success"] = false;
                $data["message"] = "please try again";
            }
        } else {
            $insert_data = array(
                'name'        => $this->input->post('name'),
                'email'       => $this->input->post('email'),
                'phone'       => $this->input->post('phone'),
                'cf1'         => $this->input->post('cf1'),
                'endereco'    => $this->input->post('bairro'),
                'numero'      => $this->input->post('numero'),
                'bairro'      => $this->input->post('bairro'),
                'cidade'      => $this->input->post('cidade'),
                'estado'      => $this->input->post('estado'),
                'data'        => date("Y-m-d"),
                'AccountID'   => 0,
                'Country'     => $this->input->post('Country'),
                'complemento' => $this->input->post('bairro'),
                'cep'         => $this->input->post('cep'),
            );

            $exist_nif = $this->customers_model->getCustomerByCf1($this->input->post('cf1'));
            if ($exist_nif) {
                $data["success"] = false;
                $data["message"] = "Este NIF já está a ser utilizado por outro cliente";
                echo json_encode($data);
                die();
            }

            if ($this->customers_model->addCustomer($insert_data)) {
                $data["success"] = true;
                $data["message"] = lang('customer_added');
            } else {
                $data["success"] = false;
                $data["message"] = "please try again";
            }
        }

        echo json_encode($data);
    }

    function delete_customer() {
        $id = $this->input->post('id');

        if ($this->customers_model->isCustomerInSales($id)) {
            $data["success"] = false;
            $data["message"] = "Lamento! mas não pode eliminar clientes com movimetos.";
            echo json_encode($data);
            die();
        } else {
            if ($this->customers_model->deleteCustomer($id)) {
                $data["success"] = true;
                $data["message"] = lang("customer_deleted");
            } else {
                $data["success"] = false;
                $data["message"] = "please try again";
            }
            echo json_encode($data);
        }
    }

    /* =========================================================
     *  FORNECEDORES
     * =======================================================*/

    function suppliers() {
        $suppliers = get_rows("tec_suppliers", array(), "name asc");
        $data      = array(
            "success"   => true,
            "suppliers" => $suppliers,
        );
        echo json_encode($data);
    }

    function get_supplier($id) {
        $supplier = $this->suppliers_model->getSupplierByID($id);
        $data     = array(
            "success"  => true,
            "message"  => "Got Data",
            "supplier" => $supplier,
        );
        echo json_encode($data);
    }

    function save_supplier() {
        $id = $this->input->post('id');

        if ($id) {
            $update_data = array(
                'name'      => $this->input->post('name'),
                'email'     => $this->input->post('email'),
                'phone'     => $this->input->post('phone'),
                'cf1'       => $this->input->post('cf1'),
                'cf2'       => $this->input->post('cf2'),
                'endereco'  => $this->input->post('endereco'),
                'numero'    => $this->input->post('numero'),
                'bairro'    => $this->input->post('bairro'),
                'cidade'    => $this->input->post('cidade'),
                'estado'    => $this->input->post('estado'),
                'AccountID' => $this->input->post('AccountID'),
                'Country'   => $this->input->post('Country'),
            );

            if ($this->suppliers_model->updateSupplier($id, $update_data)) {
                $data["success"] = true;
                $data["message"] = lang('supplier_updated');
            } else {
                $data["success"] = false;
                $data["message"] = "please try again";
            }
        } else {
            $insert_data = array(
                'name'        => $this->input->post('name'),
                'email'       => $this->input->post('email'),
                'phone'       => $this->input->post('phone'),
                'cf1'         => $this->input->post('cf1'),
                'cf2'         => " ",
                'endereco'    => $this->input->post('endereco'),
                'Country'     => $this->input->post('Country'),
                'numero'      => $this->input->post('numero'),
                'bairro'      => $this->input->post('bairro'),
                'cidade'      => $this->input->post('cidade'),
                'estado'      => $this->input->post('estado'),
                'AccountID'   => $this->input->post('AccountID'),
                'complemento' => " ",
                'cep'         => " ",
            );

            if ($this->suppliers_model->addSupplier($insert_data)) {
                $data["success"] = true;
                $data["message"] = lang('supplier_added');
            } else {
                $data["success"] = false;
                $data["message"] = "please try again";
            }
        }

        echo json_encode($data);
    }

    function delete_supplier() {
        $id = $this->input->post('id');

        if ($this->suppliers_model->deleteSupplier($id)) {
            $data["success"] = true;
            $data["message"] = lang("supplier_deleted");
        } else {
            $data["success"] = false;
            $data["message"] = "please try again";
        }
        echo json_encode($data);
    }

    /* =========================================================
     *  PRODUTOS LISTA / ADICIONAIS
     * =======================================================*/

    function product_items() {
        $product_items = get_rows_join(
            "tec_producs_list",
            array(),
            "",
            array(),
            "tec_products.code,
             tec_products.image,
             tec_products.name as pname,
             tec_producs_list.*",
            array(),
            array(array("table" => "tec_products", "where" => "tec_products.id=tec_producs_list.product"))
        );

        $data = array(
            "success"       => true,
            "product_items" => $product_items,
        );
        echo json_encode($data);
    }

    function get_product_item($id) {
        $product_item = get_row("tec_producs_list", array("id" => $id));
        $data         = array(
            "success"      => true,
            "message"      => "Got Data",
            "product_item" => $product_item,
        );
        echo json_encode($data);
    }

    function save_product_item() {
        $id          = $this->input->post('id');
        $insert_data = array(
            'product' => $this->input->post('product'),
            'name'    => $this->input->post('name'),
            'price'   => $this->input->post('price'),
        );

        if ($id) {
            update_row("tec_producs_list", $insert_data, array("id" => $id));
            $data["success"] = true;
            $data["message"] = "Adicional alterad com sucesso";
        } else {
            create_row("tec_producs_list", $insert_data);
            $data["success"] = true;
            $data["message"] = "Novo adicional, adicionado com sucesso";
        }
        echo json_encode($data);
    }

    function delete_product_item() {
        $id = $this->input->post('id');

        if (delete_row("tec_producs_list", array("id" => $id))) {
            $data["success"] = true;
            $data["message"] = lang("Adicional deletado com sucesso");
        } else {
            $data["success"] = false;
            $data["message"] = "please try again";
        }
        echo json_encode($data);
    }

    /* =========================================================
     *  PRODUTOS
     * =======================================================*/

    function products() {
        $products = get_rows_left_join(
            "tec_products",
            array(),
            "name asc",
            array(),
            "tec_products.*",
            array(),
            array(array("table" => "tec_categories", "where" => "tec_products.category_id=tec_categories.id"))
        );

        $data = array(
            "success" => true,
            "message" => "Got Data",
            "data"    => $products,
        );
        echo json_encode($data);
    }

    function get_product($id) {
        $product              = $this->site->getProductByID($id);
        $product->type_text   = lang($product->type);
        $category             = $this->site->getCategoryByID($product->category_id);
        $combo_items          = $product->type == 'combo'
            ? $this->products_model->getComboItemsByPID($id)
            : NULL;

        $data = array(
            "success"     => true,
            "message"     => "Got Data",
            "product"     => $product,
            "category"    => $category,
            "combo_items" => $combo_items,
        );
        echo json_encode($data);
    }

    function save_product() {
        // (mantido o mesmo comportamento que o original)
        $id          = $this->input->post('id');
        $insert_data = array(
            'product' => $this->input->post('product'),
            'name'    => $this->input->post('name'),
            'price'   => $this->input->post('price'),
        );

        if ($id) {
            update_row("tec_producs_list", $insert_data, array("id" => $id));
            $data["success"] = true;
            $data["message"] = "Adicional alterad com sucesso";
        } else {
            create_row("tec_producs_list", $insert_data);
            $data["success"] = true;
            $data["message"] = "Novo adicional, adicionado com sucesso";
        }
        echo json_encode($data);
    }

    function delete_product() {
        $id = $this->input->post('id');

        if (delete_row("tec_producs_list", array("id" => $id))) {
            $data["success"] = true;
            $data["message"] = lang("Adicional deletado com sucesso");
        } else {
            $data["success"] = false;
            $data["message"] = "please try again";
        }
        echo json_encode($data);
    }

    /* =========================================================
     *  TAXAS
     * =======================================================*/

    function taxes() {
        $taxes = get_rows("tec_tax", array(), "code asc");
        $data  = array(
            "success" => true,
            "message" => "Got Data",
            "taxes"   => $taxes,
        );
        echo json_encode($data);
    }

    
    /* =========================================================
     *  LOGIN (esqueleto)
     * =======================================================*/
    public function login()
    {
        // Use configured AGT credentials from `tec_settings` (loaded in constructor).
        $username = isset($this->minfin_user) ? $this->minfin_user : '';
        $password = isset($this->minfin_pass) ? $this->minfin_pass : '';

        // implementação do login se necessário
    }

    /**
     * Chamada interna à API MinFin/AGT (SIGT) para obter dados do contribuinte
     * HML: https://sifphml.minfin.gov.ao/sigt/contribuinte/v5/obter?tipoDocumento=NIF&numeroDocumento=XXXXXXXXXXX
     *
     * @param string $nif
     * @return array  ['success'=>true,'data'=>[...] ] OU ['success'=>false,'error'=>...]
     */
  /**
 * Chamada interna à API MinFin/AGT (SIGT) para obter dados do contribuinte
 * Suporta dois endpoints:
 * 1. SIGT consultarNIF (básico, 6 campos)
 * 2. Invoice MinFin (completo, todos os campos)
 *
 * @param string $nif
 * @param string $endpoint 'sigt' (padrão) ou 'invoice'
 * @return array  ['success'=>true,'data'=>[...] ] OU ['success'=>false,'error'=>...]
 */
private function callMinfinTaxpayer($nif, $endpoint = 'sigt')
{
    $nif = trim((string)$nif);

    if ($nif === '') {
        return array(
            'success' => false,
            'error'   => 'NIF / número de documento vazio.'
        );
    }

    // ========== ENDPOINT 1: SIGT (básico) ==========
    if ($endpoint === 'sigt') {
        $base = isset($this->minfin_base) && $this->minfin_base ? $this->minfin_base : 'https://sifp.minfin.gov.ao/sigt';
        $baseUrl = rtrim($base, '/') . '/contribuinte/consultarNIF/v5/obter';
        $query   = http_build_query(array(
            'tipoDocumento'   => 'NIF',
            'numeroDocumento' => $nif,
        ));
        $url = $baseUrl . '?' . $query;
        $useBasicAuth = true;
    }
    // ========== ENDPOINT 2: Invoice MinFin (completo) ==========
    else {
        $url = 'https://invoice.minfin.gov.ao/commonServer/common/taxpayer/get/' . $nif;
        $useBasicAuth = false; // Testar sem autenticação primeiro
    }

    $ch = curl_init($url);
    
    $curlOpts = array(
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_FOLLOWLOCATION => true,
        CURLOPT_CONNECTTIMEOUT => 8,
        CURLOPT_TIMEOUT        => 15,
        CURLOPT_SSL_VERIFYHOST => 0,
        CURLOPT_SSL_VERIFYPEER => false,
        CURLOPT_USERAGENT      => 'WPV-Modelo/CI3 (+SIF-POS)',
        CURLOPT_HTTPHEADER     => array(
            'Accept: application/json',
            'Accept-Charset: utf-8',
        ),
    );
    
    // Adiciona autenticação apenas para SIGT
    if ($useBasicAuth) {
        $curlOpts[CURLOPT_HTTPAUTH] = CURLAUTH_BASIC;
        $curlOpts[CURLOPT_USERPWD] = $this->minfin_user . ':' . $this->minfin_pass;
    }
    
    curl_setopt_array($ch, $curlOpts);

    $resp     = curl_exec($ch);
    $curl_err = curl_error($ch);
    $code     = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);

    // Erro de transporte (DNS, timeout, etc.)
    if ($resp === false) {
        return array(
            'success' => false,
            'error'   => 'Erro ao contactar API MinFin (SIGT): ' . $curl_err,
        );
    }

    // 401 / Unauthorized
    if ($code == 401 || stripos($resp, 'Unauthorized') !== false) {
        return array(
            'success' => false,
            'error'   => 'Não autorizado na API MinFin (SIGT). Verifique utilizador/senha.',
            'raw'     => $resp,
            'code'    => $code,
        );
    }

    $json = json_decode($resp, true);

    if (json_last_error() !== JSON_ERROR_NONE || !is_array($json)) {
        return array(
            'success' => false,
            'error'   => 'Resposta inválida da API MinFin (' . $endpoint . ').',
            'raw'     => $resp,
            'code'    => $code,
        );
    }

    // ========== PARSING ENDPOINT SIGT ==========
    if ($endpoint === 'sigt') {
        // Estrutura: { "ObterContribuinte": { "mensagem": "...", "contribuinte": { ... } } }
        if (!empty($json['ObterContribuinte']) && is_array($json['ObterContribuinte'])) {
            $obter = $json['ObterContribuinte'];

            if (!empty($obter['contribuinte']) && is_array($obter['contribuinte'])) {
                return array(
                    'success' => true,
                    'data'    => $obter['contribuinte'],
                    'code'    => $code,
                    'raw'     => $json,
                );
            }

            $mensagem = !empty($obter['mensagem']) ? $obter['mensagem'] : 'Contribuinte não encontrado.';
            return array(
                'success' => false,
                'error'   => $mensagem,
                'raw'     => $json,
                'code'    => $code,
            );
        }
    }
    // ========== PARSING ENDPOINT INVOICE ==========
    else {
        // Estrutura esperada: objeto direto com dados do contribuinte
        // ou { "success": true, "data": {...} }
        if (!empty($json['data']) && is_array($json['data'])) {
            return array(
                'success' => true,
                'data'    => $json['data'],
                'code'    => $code,
                'raw'     => $json,
            );
        } elseif (isset($json['nif']) || isset($json['numeroNIF']) || isset($json['name']) || isset($json['nome'])) {
            // Dados vêm direto no root
            return array(
                'success' => true,
                'data'    => $json,
                'code'    => $code,
                'raw'     => $json,
            );
        }
    }

    return array(
        'success' => false,
        'error'   => 'Formato inesperado na resposta (' . $endpoint . ').',
        'raw'     => $json,
        'code'    => $code,
    );
}


    /**
     * Tenta inferir cidade/província a partir do endereço,
     * procurando pelo nome das províncias de Angola.
     */
    private function guessLocationFromAddress($address)
    {
        $address = (string) $address;
        if ($address === '') {
            return array('city' => '', 'state' => '');
        }

        $provinces = array(
            'Luanda','Benguela','Huíla','Huila','Huambo','Bié','Bie','Malanje','Malange',
            'Cuanza Sul','Kwanza Sul','Cuanza Norte','Kwanza Norte',
            'Namibe','Cabinda','Zaire','Cunene','Cuando Cubango',
            'Uíge','Uige','Bengo','Lunda Norte','Lunda Sul','Moxico'
        );

        $upper = mb_strtoupper($address, 'UTF-8');
        foreach ($provinces as $prov) {
            $pUpper = mb_strtoupper($prov, 'UTF-8');
            if (mb_strpos($upper, $pUpper) !== false) {
                return array('city' => $prov, 'state' => $prov);
            }
        }

        return array('city' => '', 'state' => '');
    }

    /**
     * Mapeia códigos de repartição fiscal (ssswjg) para províncias de Angola.
     * Ex: "11.01" (RF BENGUELA) -> "Benguela"
     */
    private function mapReparticaoFiscal($codigo)
    {
        $codigo = trim((string)$codigo);
        if ($codigo === '') {
            return '';
        }

        // Mapa de repartições fiscais para províncias
        $map = array(
            // Cabinda
            '01.01' => 'Cabinda',
            '01.02' => 'Cabinda',
            
            // Luanda
            '02.01' => 'Luanda',
            '02.02' => 'Luanda',
            '02.03' => 'Luanda',
            
            // Uíge
            '03.01' => 'Uíge',
            
            // Repartições Fiscais (4ª série)
            '04.01' => 'Luanda',
            '04.02' => 'Luanda',
            '04.03' => 'Luanda',
            '04.05' => 'Luanda',
            '04.10' => 'Luanda',
            '04.102' => 'Luanda',
            '04.11' => 'Luanda',
            '04.20' => 'Luanda',
            '04.33' => 'Luanda',
            '04.507' => 'Luanda',
            
            // Ndalatando
            '05.01' => 'Cuanza Norte',
            '05.02' => 'Cuanza Norte',
            '05.03' => 'Cuanza Norte',
            
            // Sumbe
            '06.01' => 'Cuanza Sul',
            '06.02' => 'Cuanza Sul',
            '06.03' => 'Cuanza Sul',
            '06.04' => 'Cuanza Sul',
            '06.05' => 'Cuanza Sul',
            '06.06' => 'Cuanza Sul',
            
            // Malanje
            '07.01' => 'Malanje',
            
            // Chitato
            '08.01' => 'Lunda Norte',
            
            // Saurimo
            '09.01' => 'Lunda Sul',
            
            // Benguela
            '11.01' => 'Benguela',
            '11.02' => 'Benguela',
            '11.03' => 'Benguela',
            '11.04' => 'Benguela',
            
            // Huambo
            '12.01' => 'Huambo',
            '12.02' => 'Huambo',
            '12.03' => 'Huambo',
            
            // Cuito
            '13.01' => 'Bié',
            '13.02' => 'Bié',
            '13.03' => 'Bié',
            '13.04' => 'Bié',
            
            // Luena
            '14.01' => 'Moxico',
            '14.02' => 'Moxico',
            
            // Calai
            '15.008' => 'Cuando Cubango',
            '15.01' => 'Cuando Cubango',
            
            // Menongue
            '16.01' => 'Cuando Cubango',
            '16.02' => 'Cuando Cubango',
            
            // Lubango
            '17.01' => 'Huíla',
            '17.05' => 'Huíla',
            
            // Xangongo
            '18.01' => 'Cunene',
            '18.02' => 'Cunene',
            
            // Caxito
            '19.01' => 'Bengo',
            '19.03' => 'Bengo',
        );

        // Retorna a província mapeada
        if (isset($map[$codigo])) {
            return $map[$codigo];
        }

        // Se não encontrar mapeamento exato, retorna vazio
        return '';
    }

    /**
     * Mapeia códigos de província (ex: "LA") para nomes completos (ex: "Luanda").
     */
    private function mapProvinceCode($code)
    {
        $code = strtoupper(trim((string)$code));
        if ($code === '') {
            return '';
        }

        $map = array(
            'LA' => 'Luanda',
            'BE' => 'Benguela',
            'BA' => 'Benguela',
            'BG' => 'Bengo',
            'BI' => 'Bié',
            'CA' => 'Cabinda',
            'CC' => 'Cuando Cubango',
            'CN' => 'Cuanza Norte',
            'CS' => 'Cuanza Sul',
            'CU' => 'Cunene',
            'HU' => 'Huambo',
            'HL' => 'Huíla',
            'LN' => 'Lunda Norte',
            'LS' => 'Lunda Sul',
            'MA' => 'Malanje',
            'MO' => 'Moxico',
            'NA' => 'Namibe',
            'UI' => 'Uíge',
            'ZA' => 'Zaire',
        );

        if (isset($map[$code])) {
            return $map[$code];
        }

        // Se já vier por extenso, devolve como está
        return $code;
    }

    /**
     * GET /api/taxpayer?nif=XXXXXXXXX
     * Proxy "bruto" para a API do MinFin (SIGT). Mantido para debug.
     */
    public function taxpayer()
    {
        $raw = $this->input->get('nif', true);
        if (!$raw) {
            $raw = $this->uri->segment(3);
        }

       $nif = trim((string)$raw);

        if (strlen($nif) < 9) {
            return $this->output
                ->set_status_header(400)
                ->set_content_type('application/json', 'utf-8')
                ->set_output(json_encode(array(
                    'success' => false,
                    'error'   => 'NIF inválido.',
                )));
        }

        // Consumidor final (regra interna do sistema)
        if ($nif === '999999999') {
            return $this->output
                ->set_status_header(200)
                ->set_content_type('application/json', 'utf-8')
                ->set_output(json_encode(array(
                    'success' => true,
                    'data'    => array(
                        'nif'         => $nif,
                        'companyName' => 'Consumidor final',
                        'gsmc'        => 'Consumidor final',
                    ),
                )));
        }

        $result = $this->callMinfinTaxpayer($nif);

        if (!$result['success']) {
            return $this->output
                ->set_status_header(502)
                ->set_content_type('application/json', 'utf-8')
                ->set_output(json_encode($result));
        }

        $out = array(
            'success' => true,
            'nif'     => $nif,
            'data'    => $result['data'],
            'raw'     => isset($result['raw']) ? $result['raw'] : null,
        );

        return $this->output
            ->set_status_header(200)
            ->set_content_type('application/json', 'utf-8')
            ->set_output(json_encode($out));
    }

    /**
     * Endpoint usado pelo formulário de clientes
     * GET /api/consultar-nif/{nif}
     * Retorna JSON pronto para o auto-preenchimento.
     */
    public function consultar_nif_cliente($nif = null)
    {
        // NIF pela URL ou ?nif=...
        $nif = $nif ?: $this->input->get('nif');
      $nif = trim((string)$nif);

        // 1) NIF inválido
        if (strlen($nif) < 9) {
            return $this->output
                ->set_content_type('application/json', 'utf-8')
                ->set_status_header(200)
                ->set_output(json_encode(array(
                    'success' => false,
                    'message' => 'NIF inválido.',
                )));
        }

        // 2) Consumidor final (regra interna)
        if ($nif === '999999999') {
            return $this->output
                ->set_content_type('application/json', 'utf-8')
                ->set_status_header(200)
                ->set_output(json_encode(array(
                    'success'               => true,
                    'message'               => 'Consumidor final.',
                    'nif'                   => $nif,
                    'name'                  => 'Consumidor Final',
                    'company'               => 'Consumidor Final',
                    'address'               => '',
                    'city'                  => '',
                    'state'                 => '',
                    'country'               => 'Angola',
                    'phone'                 => '',
                    'email'                 => '',
                    'cae'                   => '',
                    'status'                => 'Consumidor final',
                    'estado_contribuinte'   => '',
                    'estado_atividade'      => '',
                    'estado_inadimplencia'  => '',
                )));
        }

        // 3) ESTRATÉGIA EM CASCATA: Tenta SIGT primeiro, depois Invoice se precisar
        
        // 3.1) Primeira tentativa: SIGT (rápido, mas básico)
        $agt = $this->callMinfinTaxpayer($nif, 'sigt');
        
        if (!$agt['success']) {
            return $this->output
                ->set_content_type('application/json', 'utf-8')
                ->set_status_header(200)
                ->set_output(json_encode(array(
                    'success' => false,
                    'message' => $agt['error'],
                    'raw'     => isset($agt['raw']) ? $agt['raw'] : null,
                )));
        }

        $d = isset($agt['data']) ? $agt['data'] : array();
        
        // 3.2) Verifica se os dados estão incompletos
        $precisaSegundaConsulta = false;
        
        // Verifica campos críticos vazios
        $temEmail = !empty($d['email']) || 
                    !empty($d['ListaItemListaEmail']['ItemListaEmail'][0]['email']);
        $temTelefone = !empty($d['telefone']) || 
                       !empty($d['gsmc']) || 
                       !empty($d['contactoPrincipal']['numeroContacto']);
        $temMorada = !empty($d['moradaPrincipal']['morada']);
        $temCAE = !empty($d['ListaItemListaCAE']['ItemListaCAE'][0]['numeroCAE']) ||
                  !empty($d['cae']);
        
        // Se faltar mais de 2 campos importantes, tenta o segundo endpoint
        $camposVazios = 0;
        if (!$temEmail) $camposVazios++;
        if (!$temTelefone) $camposVazios++;
        if (!$temMorada) $camposVazios++;
        if (!$temCAE) $camposVazios++;
        
        if ($camposVazios >= 2) {
            $precisaSegundaConsulta = true;
        }
        
        // 3.3) Segunda tentativa: Invoice MinFin (completo)
        if ($precisaSegundaConsulta) {
            $invoice = $this->callMinfinTaxpayer($nif, 'invoice');
            
            if ($invoice['success'] && !empty($invoice['data'])) {
                // Mescla dados: prioriza dados do Invoice (mais completo)
                $d = array_merge($d, $invoice['data']);
                
                // Log da segunda consulta
                $this->load->helper('file');
                write_file(APPPATH . 'cache/sigt_invoice_' . $nif . '.json', 
                          json_encode($invoice['data'], JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));
            }
        }
        
        // DEBUG: Log da estrutura final mesclada (comentar em produção)
        $this->load->helper('file');
        write_file(APPPATH . 'cache/sigt_final_' . $nif . '.json', json_encode($d, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE));

        // NIF (pode vir como 'numeroNIF' ou 'nif')
        $nifRetornado = !empty($d['numeroNIF']) ? $d['numeroNIF'] : (!empty($d['nif']) ? $d['nif'] : $nif);

        // Nome / Empresa (suporta SIGT e Invoice MinFin)
        $nome = '';
        if (!empty($d['companyName'])) {
            // Invoice MinFin
            $nome = $d['companyName'];
        } elseif (!empty($d['gsmc'])) {
            // Invoice MinFin (campo alternativo)
            $nome = $d['gsmc'];
        } elseif (!empty($d['nome'])) {
            // SIGT
            $nome = $d['nome'];
        } elseif (!empty($d['nomeColectivo'])) {
            $nome = $d['nomeColectivo'];
        } elseif (!empty($d['designacaoComercial'])) {
            $nome = $d['designacaoComercial'];
        } elseif (!empty($d['nomeContribuinte'])) {
            $nome = $d['nomeContribuinte'];
        }

        // Tipo de Contribuinte: COLLECTIVE, SINGULAR, etc.
        $tipoContribuinte = !empty($d['tipoContribuinte']) ? $d['tipoContribuinte'] : '';
        
        // Regime IVA: EXCL (Excluído), NORMAL, etc.
        $regimeIva = !empty($d['regimeIva']) ? $d['regimeIva'] : '';
        
        // Indicador Não Residente: F (Falso), V (Verdadeiro)
        $naoResidente = !empty($d['indicadorNaoResidente']) ? $d['indicadorNaoResidente'] : '';

                // --- TELEFONE & EMAIL PRINCIPAIS -----------------------------------

        // Dados principais (padrão)
        $telefone = '';
        $email    = '';

        // Invoice MinFin: campos específicos
        
        // lxfs: campo principal de telefone no Invoice MinFin (ex: "+244929319262")
        if (!empty($d['lxfs'])) {
            $lxfsValue = trim((string)$d['lxfs']);
            // Remove +, espaços, pontos, hífens e código do país 244
            $lxfsValue = str_replace(array('+244', '+', ' ', '.', '-'), '', $lxfsValue);
            if (preg_match('/^\d{9,}$/', $lxfsValue)) { // Se for numérico com 9+ dígitos
                $telefone = $lxfsValue;
            }
        }
        
        // email: pode conter telefone ou email
        if (!empty($d['email']) && $telefone === '') {
            $emailValue = trim((string)$d['email']);
            // Se contém apenas números, pontos e espaços, é telefone
            if (preg_match('/^[\d.\s]+$/', $emailValue)) {
                $telefone = str_replace(array(' ', '.'), '', $emailValue); // Remove espaços e pontos
            } else {
                $email = $emailValue;
            }
        }
        
        // gsmc: campo alternativo
        if (!empty($d['gsmc']) && $telefone === '') {
            $gsmcValue = trim((string)$d['gsmc']);
            if (preg_match('/^[\d.\s+]+$/', $gsmcValue)) {
                $telefone = str_replace(array('+244', '+', ' ', '.', '-'), '', $gsmcValue);
            }
        }

        // SIGT: Email principal do próprio contribuinte
        if ($email === '' && !empty($d['ListaItemListaEmail']['ItemListaEmail']) && is_array($d['ListaItemListaEmail']['ItemListaEmail'])) {
            $firstEmail = reset($d['ListaItemListaEmail']['ItemListaEmail']);
            if (!empty($firstEmail['email'])) {
                $email = trim($firstEmail['email']);
            }
        } elseif ($email === '' && !empty($d['emails']) && is_array($d['emails'])) {
            $first = reset($d['emails']);
            if (is_array($first) && !empty($first['email'])) {
                $email = trim($first['email']);
            } elseif (is_string($first)) {
                $email = trim($first);
            }
        }

        // SIGT: Telefone principal do próprio contribuinte
        if ($telefone === '' && !empty($d['contactoPrincipal']) && is_array($d['contactoPrincipal'])) {
            if (!empty($d['contactoPrincipal']['numeroContacto'])) {
                $telefone = $d['contactoPrincipal']['numeroContacto'];
            }
        } elseif ($telefone === '' && !empty($d['telefone'])) {
            $telefone = $d['telefone'];
        } elseif ($telefone === '' && !empty($d['telefones']) && is_array($d['telefones'])) {
            $ft = reset($d['telefones']);
            if (is_array($ft) && !empty($ft['numero'])) {
                $telefone = $ft['numero'];
            } elseif (is_string($ft)) {
                $telefone = $ft;
            }
        }

        // --- REPRESENTANTE LEGAL (se existir, sobrepõe os valores) -----------

        if (!empty($d['ListaItemListaReprLegal']['ItemReprLegal'])
            && is_array($d['ListaItemListaReprLegal']['ItemReprLegal'])
        ) {
            $repr = reset($d['ListaItemListaReprLegal']['ItemReprLegal']);

            // Telefone do representante legal
            if (!empty($repr['contactoRepLegal']['numeroContacto'])) {
                $telefone = $repr['contactoRepLegal']['numeroContacto'];
            }

            // Email do representante legal
            if (!empty($repr['emailRepLegal'])) {
                $email = trim($repr['emailRepLegal']);
            }
        }

        // CAE(s) - suporta SIGT e Invoice MinFin
        $cae = '';
        
        // Invoice MinFin: campo direto
        if (!empty($d['cae'])) {
            $cae = $d['cae'];
        } elseif (!empty($d['CAE'])) {
            $cae = $d['CAE'];
        }
        
        // SIGT: ListaItemListaCAE
        if ($cae === '' && !empty($d['ListaItemListaCAE']['ItemListaCAE'])
            && is_array($d['ListaItemListaCAE']['ItemListaCAE'])
        ) {
            $caeList = array();
            foreach ($d['ListaItemListaCAE']['ItemListaCAE'] as $itemCae) {
                if (!empty($itemCae['numeroCAE'])) {
                    $caeList[] = $itemCae['numeroCAE'];
                }
            }
            $caeList = array_unique($caeList);
            $cae     = implode(',', $caeList);
        }
        
        // SIGT: ListaActividadeEconomica
        if ($cae === '' && !empty($d['ListaActividadeEconomica']) && is_array($d['ListaActividadeEconomica'])) {
            $parts = array();
            foreach ($d['ListaActividadeEconomica'] as $it) {
                if (!empty($it['numeroCAE'])) $parts[] = $it['numeroCAE'];
            }
            $cae = implode(',', array_unique($parts));
        }

        // Morada principal (suporta SIGT e Invoice MinFin)
        $endereco        = '';
        $cidade          = '';
        $provincia       = '';
        $moradaPrincipal = array();

        // Invoice MinFin: campos específicos
        if (!empty($d['addressDbb'])) {
            $endereco = $d['addressDbb'];
        } elseif (!empty($d['nsrd2'])) {
            $endereco = $d['nsrd2'];
        }

        // SIGT: moradaPrincipal
        if ($endereco === '' && !empty($d['moradaPrincipal']) && is_array($d['moradaPrincipal'])) {
            $moradaPrincipal = $d['moradaPrincipal'];

            $partes = array();

            if (!empty($moradaPrincipal['morada'])) {
                $partes[] = $moradaPrincipal['morada'];
            }
            if (!empty($moradaPrincipal['bairro'])) {
                $partes[] = $moradaPrincipal['bairro'];
            }
            if (!empty($moradaPrincipal['localidade'])) {
                $partes[] = $moradaPrincipal['localidade'];
            }
            if (!empty($moradaPrincipal['municipio'])) {
                $partes[] = $moradaPrincipal['municipio'];
            }
            if (!empty($moradaPrincipal['provincia'])) {
                $partes[] = $moradaPrincipal['provincia'];
            }

            $endereco = implode(', ', $partes);

            $cidade        = !empty($moradaPrincipal['municipio']) ? $moradaPrincipal['municipio'] : '';
            $provinciaCode = !empty($moradaPrincipal['provincia']) ? $moradaPrincipal['provincia'] : '';
            $provincia     = $this->mapProvinceCode($provinciaCode);
        }

        // Invoice MinFin: ssswjg (código de repartição fiscal) - melhor fonte para província
        if (!empty($d['ssswjg'])) {
            $provinciaFromRF = $this->mapReparticaoFiscal($d['ssswjg']);
            if ($provinciaFromRF !== '') {
                $provincia = $provinciaFromRF;
                if ($cidade === '') {
                    $cidade = $provinciaFromRF; // Usa província como cidade se não tiver
                }
            }
        }

        // Se ainda não tivermos cidade/província, tenta pelo texto
        if (($cidade === '' || $provincia === '') && $endereco !== '') {
            $loc = $this->guessLocationFromAddress($endereco);
            if ($cidade === '') {
                $cidade = $loc['city'];
            }
            if ($provincia === '') {
                $provincia = $loc['state'];
            }
        }

        // ESTADOS
        $estadoContribuinte  = !empty($d['estadoContribuinte'])  ? $d['estadoContribuinte']  : '';
        $estadoActividade    = !empty($d['estadoActividade'])    ? $d['estadoActividade']    : '';
        $estadoInadimplencia = isset($d['estadoInadimplencia'])  ? $d['estadoInadimplencia'] : '';

        // Mapear código do estado (A=Activo, I=Inactivo, etc.)
        $estadoTexto = '';
        if ($estadoContribuinte === 'A' || $estadoContribuinte === 'ACTIVO') {
            $estadoTexto = 'Activo';
        } elseif ($estadoContribuinte === 'I' || $estadoContribuinte === 'INACTIVO') {
            $estadoTexto = 'Inactivo';
        } elseif ($estadoContribuinte !== '') {
            $estadoTexto = $estadoContribuinte;
        }

        $statusParts = array();
        if ($estadoTexto !== '') {
            $statusParts[] = 'Estado: ' . $estadoTexto;
        }
        if ($tipoContribuinte !== '') {
            $tipoTexto = ($tipoContribuinte === 'COLLECTIVE') ? 'Colectivo' : (($tipoContribuinte === 'SINGULAR') ? 'Singular' : $tipoContribuinte);
            $statusParts[] = 'Tipo: ' . $tipoTexto;
        }
        if ($regimeIva !== '') {
            $ivaTexto = ($regimeIva === 'EXCL') ? 'IVA Excluído' : 'IVA ' . $regimeIva;
            $statusParts[] = $ivaTexto;
        }
        if ($estadoActividade !== '') {
            $statusParts[] = 'Atividade: ' . $estadoActividade;
        }
        if ($estadoInadimplencia !== '') {
            $val = strtolower((string)$estadoInadimplencia);
            $ina = ($val === 'true' || $val === '1' || $val === 'v');
            $statusParts[] = 'Inadimplência: ' . ($ina ? 'Inadimplente' : 'Regular');
        }
        $status = implode(' | ', $statusParts);

        // Mensagem informativa sobre campos disponíveis
        $avisoMessage = 'Contribuinte encontrado.';
        
        $camposFaltantes = array();
        if ($telefone === '') $camposFaltantes[] = 'telefone';
        if ($email === '') $camposFaltantes[] = 'email';
        if ($endereco === '') $camposFaltantes[] = 'morada';
        if ($cae === '') $camposFaltantes[] = 'CAE';
        
        if (!empty($camposFaltantes)) {
            $avisoMessage .= ' Preencha manualmente: ' . implode(', ', $camposFaltantes) . '.';
        }

        $result = array(
            'success'               => true,
            'message'               => $avisoMessage,
            'nif'                   => $nifRetornado,
            'name'                  => $nome,
            'company'               => $nome,
            'address'               => $endereco,
            'city'                  => $cidade,
            'state'                 => $provincia,
            'country'               => 'Angola',
            'phone'                 => $telefone,
            'email'                 => $email,
            'cae'                   => $cae,
            'status'                => $status,
            'estado_contribuinte'   => $estadoTexto,
            'estado_atividade'      => $estadoActividade,
            'estado_inadimplencia'  => $estadoInadimplencia,
            'tipo_contribuinte'     => $tipoContribuinte,
            'regime_iva'            => $regimeIva,
            'nao_residente'         => $naoResidente,
        );

        return $this->output
            ->set_content_type('application/json', 'utf-8')
            ->set_status_header(200)
            ->set_output(json_encode($result));
    }


    

    // /Api/agt_send/{sale_id}
    public function agt_send($sale_id = null)
    {
        if (!$sale_id) {
            return $this->_json(['ok' => false, 'error' => 'sale_id em falta'], 400);
        }

        // chama o controller Agt como request interno
        $_GET['send'] = 1;
        $_GET['debug'] = 0;
        $this->load->library('router'); // garante CI ok
        require_once(APPPATH . 'controllers/Agt.php');
        $agt = new Agt();
        $agt->registar_factura($sale_id);
    }

    // /Api/agt_update_status/{sale_id}
    public function agt_update_status($sale_id = null)
    {
        if (!$sale_id) {
            return $this->_json(['ok' => false, 'error' => 'sale_id em falta'], 400);
        }

        $_GET['send'] = 1;
        $_GET['debug'] = 0;
        require_once(APPPATH . 'controllers/Agt.php');
        $agt = new Agt();
        $agt->obter_estado($sale_id);
    }

    private function _json($arr, $status = 200)
    {
        $this->output
            ->set_status_header($status)
            ->set_content_type('application/json', 'utf-8')
            ->set_output(json_encode($arr, JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT))
            ->_display();
        exit;
    }

    /**
     * POST JSON com BasicAuth (PHP 5.6)
     */
    public static function post_json_basic($url, $payloadArr, $username, $password, $timeout = 60)
    {
        $body = json_encode($payloadArr, JSON_UNESCAPED_SLASHES);

        $headers = array(
            'Accept: application/json',
            'Content-Type: application/json; charset=utf-8',
            'Content-Length: ' . strlen($body),
        );

        $ch = curl_init($url);
        curl_setopt_array($ch, array(
            CURLOPT_POST           => true,
            CURLOPT_POSTFIELDS     => $body,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_HTTPAUTH       => CURLAUTH_BASIC,
            CURLOPT_USERPWD        => $username . ':' . $password,
            CURLOPT_HTTPHEADER     => $headers,
            CURLOPT_TIMEOUT        => $timeout,
            CURLOPT_CONNECTTIMEOUT => 20,
            CURLOPT_SSL_VERIFYPEER => false,
            CURLOPT_SSL_VERIFYHOST => false,
        ));

        $respBody = curl_exec($ch);
        $code     = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $err      = curl_error($ch);
        curl_close($ch);

        return array(
            'ok'        => ($code >= 200 && $code < 300),
            'http_code' => $code,
            'body'      => $respBody,
            'error'     => $err,
        );
    }
    
}


